import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { ImageStore, ImageUpload } from "aegis/features/images";
import { Display, JustifyContent, Layout } from "twitch-core-ui";
import { UploadingImageThumbnail } from "./image-thumbnail";
import { ImageViewModal } from "./viewer-modal";

import "./styles.scss";

export interface PublicProps {
  store: ImageStore;
  fullComments?: boolean;
  hideComments?: boolean;
  excludedImagesURLs?: string[];
  imageUploaded?: (imageURLComment: string) => void;
}
@observer
export class ImageDisplay extends React.Component<PublicProps> {
  @observable viewerModalVisible: boolean = false;

  public render() {
    const { store, excludedImagesURLs } = this.props;
    const renderedImages = store.images.map((image, index) => {
      const { imageURL, error } = image;
      const excluded = excludedImagesURLs && this.props.excludedImagesURLs!.includes(imageURL!);
      const onClick = error ? this.retryImageUpload : excluded ? this.reAddImageURL : this.openViewerModal;
      return (
        <UploadingImageThumbnail
          key={"preview-image-" + index}
          image={image}
          onClick={onClick}
          excluded={excluded}
          notUploaded={!image.imageURL && !image.uploading}
        />
      );
    });

    const viewerModal = this.viewerModalVisible ? (
      <ImageViewModal
        onClose={this.closeViewerModal}
        images={store.images}
        excludedImagesURLs={excludedImagesURLs}
        retryUpload={this.retryImageUpload}
      />
    ) : null;

    return (
      <Layout display={Display.Flex} justifyContent={JustifyContent.Between} fullWidth>
        {viewerModal}
        <Layout className="screenshot-thumbnails-maxwidth">{renderedImages}</Layout>
      </Layout>
    );
  }

  @action
  private openViewerModal = () => {
    this.viewerModalVisible = true;
  };
  @action
  private closeViewerModal = () => {
    this.viewerModalVisible = false;
  };

  private retryImageUpload = (image: ImageUpload) => {
    this.props.store.retryUpload(image.imageDataURL, this.props.imageUploaded);
  };

  private reAddImageURL = (image: ImageUpload) => {
    if (this.props.imageUploaded && image.imageURL) {
      this.props.imageUploaded(this.props.store.formatImageComment(image, image.imageURL));
    }
  };
}
