import * as React from "react";

import { ImageUpload } from "aegis/features/images/models";
import {
  AlignItems,
  Background,
  CoreImage,
  CoreText,
  Display,
  Interactable,
  InteractableType,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Position,
  SpinnerSize,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
  Tooltip
} from "twitch-core-ui";

import "./styles.scss";

export interface PublicProps {
  image: ImageUpload;
  onClick?: (image: ImageUpload) => void;
  excluded?: boolean;
  notUploaded: boolean;
}

export class UploadingImageThumbnail extends React.Component<PublicProps> {
  public render() {
    const { image } = this.props;

    let overlay: JSX.Element | null = null;
    let hoverHint: JSX.Element | null = null;
    let tooltip: string | null = null;

    if (image.error) {
      overlay = <SVG asset={SVGAsset.NotificationError} type={SVGType.Alert} height={30} width={30} />;
      hoverHint = <CoreText>Click to try again</CoreText>;
      tooltip = "Error uploading this image";
    } else if (image.uploading) {
      overlay = <LoadingSpinner size={SpinnerSize.Large} fillContent />;
      tooltip = "uploading...";
    } else if (this.props.excluded) {
      overlay = <CoreText>Excluded</CoreText>;
      hoverHint = <CoreText>Click to re add</CoreText>;
      tooltip = "You deleted the URL for this screenshot";
    } else if (this.props.notUploaded) {
      overlay = <CoreText>Not uploaded</CoreText>;
      tooltip = "This image has not been uploaded yet";
    } else {
      hoverHint = <CoreText>Click to view images</CoreText>;
    }

    const thumbnail = (
      <Layout padding={{ left: 1, bottom: 1 }} display={Display.InlineBlock}>
        <Interactable
          hover={false}
          onClick={this.handleClick}
          selected={!!image.error}
          type={(image.error && InteractableType.Alert) || undefined}
        >
          <StyledLayout position={Position.Relative} className="thumbnail">
            <CoreImage className="thumbnail__image" src={image.imageDataURL} alt="(image preview)" />
            {overlay && (
              <StyledLayout
                background={Background.Overlay}
                className="thumbnail__overlay"
                position={Position.Absolute}
                attachTop
                attachLeft
                fullWidth
                fullHeight
                display={Display.Flex}
                alignItems={AlignItems.Center}
                justifyContent={JustifyContent.Center}
              >
                {overlay}
              </StyledLayout>
            )}

            {(hoverHint || overlay) && (
              <StyledLayout
                background={Background.Overlay}
                className="thumbnail__hover-tip"
                position={Position.Absolute}
                attachTop
                attachLeft
                fullWidth
                fullHeight
                display={Display.Flex}
                alignItems={AlignItems.Center}
                justifyContent={JustifyContent.Center}
              >
                {hoverHint || overlay}
              </StyledLayout>
            )}
          </StyledLayout>
        </Interactable>
      </Layout>
    );

    return tooltip ? <Tooltip label={tooltip}>{thumbnail}</Tooltip> : thumbnail;
  }

  private handleClick = () => {
    if (this.props.onClick) {
      this.props.onClick(this.props.image);
    }
  };
}
