import * as React from "react";

import { observer } from "mobx-react";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import { ImageUpload } from "aegis/features/images";
import { CopyLinkButton } from "aegis/functionality/components/copy-link-button";
import { StackedModal } from "aegis/functionality/components/stacked-modal";

import {
  AlignContent,
  AlignItems,
  Background,
  Button,
  ButtonType,
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  InFeatureNotification,
  InFeatureNotificationType,
  JustifyContent,
  Layout,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
  TextType,
  Tooltip,
  TooltipDirection
} from "twitch-core-ui";

import "./styles.scss";

export interface PublicProps {
  onClose: () => void;
  images: ImageUpload[];
  excludedImagesURLs?: string[];
  retryUpload: (image: ImageUpload) => void;
}

interface PreviewImageProps {
  image: ImageUpload;
  excluded: boolean;
  retryUpload: (image: ImageUpload) => void;
}

const PreviewImage: React.SFC<PreviewImageProps> = ({ image, excluded, retryUpload }) => {
  const { imageComment, imageDataURL, imageURL, error } = image;
  const comment = imageComment ? (
    <Layout display={Display.Flex} padding={{ bottom: 1 }}>
      <Layout padding={{ right: 1 }}>
        <StyledLayout color={Color.Alt2}>
          <SVG asset={SVGAsset.Conversations} />
        </StyledLayout>
      </Layout>
      <CoreText fontSize={FontSize.Size5}>{imageComment}</CoreText>
    </Layout>
  ) : null;

  const link = (
    <Layout display={Display.Flex}>
      <Layout padding={{ right: 1 }}>
        <Tooltip direction={TooltipDirection.Right} label="Copy to clipboard (cmd+click to open in new tab)">
          <CopyLinkButton link={imageURL!}>
            <StyledLayout color={Color.Alt2}>
              <SVG asset={SVGAsset.Link} />
            </StyledLayout>
          </CopyLinkButton>
        </Tooltip>
      </Layout>
      <CoreText fontSize={FontSize.Size5}>
        <CoreLink to={imageURL} targetBlank>
          {imageURL}
        </CoreLink>
      </CoreText>
    </Layout>
  );

  const errorNotification = error ? (
    <InFeatureNotification
      type={InFeatureNotificationType.Error}
      title="Failed to upload image"
      description={error}
      link={{
        children: <div>Click here to retry</div>,
        onClick: () => retryUpload(image)
      }}
    />
  ) : null;

  const imageClass = excluded ? "comment-photo excluded-thumbnail" : "comment-photo";
  const excludedHint = excluded ? (
    <div className="excluded-hint">
      <div>Excluded</div>
    </div>
  ) : null;
  return (
    <Layout
      margin={{ y: 1 }}
      className="image-view-thumbnail"
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      alignItems={AlignItems.Center}
    >
      <div className="thumbnail-container">
        <img src={imageDataURL} className={imageClass} />
        {excludedHint}
      </div>
      <Layout fullWidth padding={{ y: 1 }}>
        {errorNotification}
        {comment}
        {!error ? link : null}
      </Layout>
    </Layout>
  );
};

@observer
export class ImageViewModal extends React.Component<PublicProps> {
  public render() {
    const { excludedImagesURLs, retryUpload } = this.props;
    const renderedImages = this.props.images.map((image, index) => {
      return (
        <PreviewImage
          key={"preview-image:" + index}
          image={image}
          excluded={!!excludedImagesURLs && excludedImagesURLs.includes(image.imageURL!)}
          retryUpload={retryUpload}
        />
      );
    });

    return (
      <StackedModal onClose={this.props.onClose} title="View images">
        <AncestorNameContextProvider ancestorName={AncestorName.ScreenshotModal}>
          <StyledLayout
            className="image-view-modal"
            alignContent={AlignContent.Center}
            alignItems={AlignItems.Center}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
          >
            <StyledLayout
              className="image-view-modal-body"
              alignContent={AlignContent.Center}
              padding={1}
              fullWidth
              fullHeight
            >
              <Layout
                fullWidth
                alignItems={AlignItems.Center}
                display={Display.Flex}
                flexDirection={FlexDirection.Column}
              >
                {renderedImages}
              </Layout>
            </StyledLayout>
            <Layout className="image-view-modal-footer" fullWidth flexGrow={0} flexShrink={0}>
              <StyledLayout
                display={Display.Flex}
                justifyContent={JustifyContent.Between}
                alignItems={AlignItems.Center}
                fullWidth
                background={Background.Alt2}
                padding={1}
                attachBottom
                position={Position.Absolute}
              >
                <CoreText type={TextType.H4} color={Color.Alt2}>
                  {`${renderedImages.length} images uploaded`}
                </CoreText>
                <Layout display={Display.Flex}>
                  <Button
                    data-track-click="iamge-view-modal-close"
                    type={ButtonType.Hollow}
                    onClick={this.props.onClose}
                  >
                    {"Close"}
                  </Button>
                </Layout>
              </StyledLayout>
            </Layout>
          </StyledLayout>
        </AncestorNameContextProvider>
      </StackedModal>
    );
  }
}
