import * as React from "react";

import { action, observable } from "mobx";
import { inject, observer } from "mobx-react";

import { AncestorNameContext, AncestorNameContextMapper, withAncestorNameContext } from "aegis/context/ancestor";
import { ImageStore, ImageUpload } from "aegis/features/images";
import { EventName, Screenshot } from "aegis/features/tracking";
import { TrackingClientProps } from "aegis/features/tracking/client";
import { ImageFriendlyInputForm } from "./components/input-form";
import { ImagePreviewModal } from "./components/preview-modal";
import { ReuploadHint } from "./components/reupload-hint";

export interface PublicProps {
  store: ImageStore;
  imageUploaded?: (imageURLComment: string) => void;
  targetUserID?: string;
  reportID?: string;
}

type Props = PublicProps & TrackingClientProps & AncestorNameContext;

@inject("trackingClient")
@observer
export class ImageUploaderComponent extends React.Component<Props> {
  @observable previewFormVisible: boolean = false;
  @observable showReuploadHint: boolean = false;
  @observable images: ImageUpload[] = [];

  public constructor(props: Props) {
    super(props);
    if (props.targetUserID) {
      this.props.store.setTargetUserID(props.targetUserID);
    }
  }

  public componentWillReceiveProps(newProps: Props) {
    if (newProps.targetUserID && newProps.targetUserID !== this.props.targetUserID) {
      this.props.store.setTargetUserID(newProps.targetUserID);
    }
  }

  public render() {
    const previewModal = this.previewFormVisible ? (
      <ImagePreviewModal
        onClose={this.closePreviewModal}
        removeImage={this.removeImage}
        updateComment={this.updateImageComment}
        images={this.images}
        onPaste={this.receiveImage}
        onCancel={this.cancelUpload}
        onStartUpload={this.moveImagesAndStartUpload}
      />
    ) : null;

    const reuploadHint = this.showReuploadHint ? (
      <ReuploadHint canReupload={this.props.store.hasUserID} doReupload={this.startUpload} />
    ) : null;
    return (
      <div>
        {previewModal}
        <ImageFriendlyInputForm
          onPasteImage={this.receiveImage}
          error={this.showReuploadHint}
          openPreview={this.showPreviewModal}
        />
        {reuploadHint}
      </div>
    );
  }

  @action
  private receiveImage = (imageFile: File, imageData: string) => {
    this.previewFormVisible = true;
    this.images.push({ imageFile: imageFile, imageDataURL: imageData });
  };

  @action
  private removeImage = (imageToDelete: string) => {
    const index = this.images.findIndex(image => {
      return image.imageDataURL === imageToDelete;
    });
    if (index !== -1) {
      this.images.splice(index, 1);
    }
  };

  @action
  private updateImageComment = (imageDataURLIn: string, imageCommentIn: string) => {
    const index = this.images.findIndex(({ imageDataURL }) => {
      return imageDataURL === imageDataURLIn;
    });
    if (index === -1) {
      return;
    }
    this.images[index] = { ...this.images[index], imageComment: imageCommentIn };
  };

  @action
  private showPreviewModal = () => {
    this.previewFormVisible = true;
  };

  @action
  private closePreviewModal = () => {
    this.previewFormVisible = false;
  };

  @action
  private cancelUpload = () => {
    this.previewFormVisible = false;
    this.images = [];
  };

  @action
  private addImagesToStore = () => {
    this.props.store.addUploadingImages(this.images);
    this.previewFormVisible = false;
    this.images = [];
  };

  @action
  private startUpload = () => {
    const { reportID, ancestorName, store, imageUploaded, trackingClient } = this.props;

    if (store.hasUserID) {
      this.showReuploadHint = false;
      store.startUpload(imageUploaded);
    } else {
      this.showReuploadHint = true;
    }

    const event: Screenshot = {
      report_id: reportID,
      screenshot_count: store.images.length,
      ui_context: ancestorName
    };
    trackingClient!.Track(EventName.Screenshot, event);
  };

  private moveImagesAndStartUpload = () => {
    this.addImagesToStore();
    this.startUpload();
  };
}

export const ImageUploader = withAncestorNameContext<AncestorNameContext, PublicProps>(AncestorNameContextMapper)(
  ImageUploaderComponent
);
