import * as React from "react";
import {
  Color,
  Display,
  Input,
  InputType,
  Interactable,
  InteractableType,
  Layout,
  Position,
  StyledLayout,
  SVG,
  SVGAsset
} from "twitch-core-ui";

import { errorNotification } from "aegis/functionality/error-notifications";

export interface PublicProps {
  autoFocus?: boolean;
  error?: boolean;
  onPasteImage: (imageFile: File, base64: string) => void;
  openPreview?: () => void;
}

const supportedImageTypes = ["image/png", "image/jpeg"];

type Props = PublicProps;

interface State {
  imageInputRef: HTMLInputElement | null;
}

export class ImageFriendlyInputForm extends React.Component<Props, State> {
  public constructor(props: Props) {
    super(props);
    this.state = { imageInputRef: null };
  }

  public componentDidUpdate() {
    if (this.state.imageInputRef && this.props.autoFocus) {
      this.state.imageInputRef.focus();
    }
  }

  public render() {
    return (
      <Layout display={Display.Block} position={Position.Relative}>
        <Input
          autoFocus={this.props.autoFocus}
          type={InputType.Text}
          placeholder={"Paste Image"}
          onBlur={this.onLeaveInput}
          onFocus={this.onEnterInput}
          onChange={() => undefined}
          error={this.props.error}
          refDelegate={(imageInputRef: HTMLInputElement) => {
            if (imageInputRef && imageInputRef !== this.state.imageInputRef) {
              this.setState({ imageInputRef });
            }
          }}
        />
        {this.props.openPreview && (
          <StyledLayout
            position={Position.Absolute}
            color={Color.Alt2}
            attachRight
            attachTop
            fullHeight
            display={Display.Flex}
          >
            <Interactable
              data-track-click="image-upload-show-modal"
              onClick={this.props.openPreview}
              type={InteractableType.Alpha}
            >
              <Layout fullHeight padding={{ x: 1.0 }} display={Display.Flex}>
                <SVG asset={SVGAsset.Plus} width={16} height={16} />
              </Layout>
            </Interactable>
          </StyledLayout>
        )}
      </Layout>
    );
  }

  private onPaste = (event: Event) => {
    if (event instanceof ClipboardEvent) {
      if (event.clipboardData.files.length < 1) {
        errorNotification("Unable to upload", { error: "You didn't paste any images" });
        return;
      }

      const item = event.clipboardData.files[0];
      if (!supportedImageTypes.includes(item.type)) {
        errorNotification("Unable to upload", {
          error: "The image type you pasted is unsupported. Please use pngs or jpegs"
        });
        return;
      }

      event.preventDefault();
      const reader = new FileReader();
      reader.onload = (funcEvent: ProgressEvent & { target: { result: string } }) => {
        const imageStr = funcEvent.target.result;
        this.props.onPasteImage(item, imageStr);
      };

      reader.readAsDataURL(item);
    }
  };

  private onEnterInput = () => {
    window.addEventListener("paste", this.onPaste);
  };

  private onLeaveInput = () => {
    window.removeEventListener("paste", this.onPaste);
  };
}
