import * as React from "react";

import { observer } from "mobx-react";

import { ImageUpload } from "aegis/features/images";
import { StackedModal } from "aegis/functionality/components/stacked-modal";
import {
  AlignContent,
  AlignItems,
  Background,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
  TextArea,
  TextType
} from "twitch-core-ui";
import { ImageFriendlyInputForm } from "../input-form";

import "./styles.scss";

export interface PublicProps {
  onClose: () => void;
  onPaste: (imageFile: File, base64: string) => void;
  onCancel: () => void;
  onStartUpload: () => void;
  removeImage: (base64: string) => void;
  updateComment: (imageDataURL: string, imageComment: string) => void;
  images?: ImageUpload[];
}

interface PreviewImageProps {
  image: ImageUpload;
  removeImage: (base64: string) => void;
  updateComment: (imageDataURL: string, imageComment: string) => void;
  autoFocus?: boolean;
  onKeyDown: (e: React.KeyboardEvent<HTMLTextAreaElement>) => void;
}

interface State {
  commentInputRef: HTMLTextAreaElement | null;
  hasBeenFocused: boolean;
}

class PreviewImage extends React.Component<PreviewImageProps, State> {
  public constructor(props: PreviewImageProps) {
    super(props);
    this.state = { commentInputRef: null, hasBeenFocused: false };
  }

  public componentDidUpdate() {
    if (this.state.commentInputRef && this.props.autoFocus && !this.state.hasBeenFocused) {
      this.state.commentInputRef.focus();
      this.setState({ hasBeenFocused: true });
    }
  }

  public render() {
    const { imageDataURL, imageComment } = this.props.image;
    const removeThisImage = () => {
      this.props.removeImage(imageDataURL);
    };
    const updateComment = (e: React.FormEvent<HTMLTextAreaElement>) => {
      this.props.updateComment(imageDataURL, e.currentTarget.value);
    };

    return (
      <Layout margin={{ y: 1 }} className="image-preview-thumbnail">
        <div className="remove-image-button" onClick={removeThisImage}>
          <SVG asset={SVGAsset.Trash} />
        </div>
        <img src={imageDataURL} className="comment-photo" />
        <Layout fullWidth padding={{ top: 1 }}>
          <TextArea
            autoFocus={this.props.autoFocus}
            placeholder="Add Comment"
            defaultValue={imageComment}
            onChange={updateComment}
            onKeyDown={this.props.onKeyDown}
            refDelegate={(commentInputRef: HTMLTextAreaElement) => {
              if (commentInputRef && commentInputRef !== this.state.commentInputRef) {
                this.setState({ commentInputRef });
              }
            }}
          />
        </Layout>
      </Layout>
    );
  }
}

@observer
export class ImagePreviewModal extends React.Component<PublicProps> {
  public render() {
    const embeddedImages = this.props.images ? this.props.images : [];
    const renderedImages = embeddedImages.map((image, index) => {
      return (
        <PreviewImage
          key={"preview-image:" + index}
          autoFocus={index === embeddedImages.length - 1}
          image={image}
          removeImage={this.props.removeImage}
          updateComment={this.props.updateComment}
          onKeyDown={this.onKeyDown}
        />
      );
    });

    const noImagesPlaceholder = (
      <Layout flexGrow={1} flexShrink={1} alignItems={AlignItems.Center} display={Display.Flex}>
        <SVG asset={SVGAsset.Picture} width={200} height={200} type={SVGType.Placeholder} />
      </Layout>
    );

    return (
      <StackedModal onClose={this.props.onClose} title="Preview or add more images">
        <StyledLayout
          className="image-preview-modal"
          alignContent={AlignContent.Center}
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
        >
          <StyledLayout
            className="image-preview-modal-body"
            alignContent={AlignContent.Center}
            padding={{ y: 1 }}
            fullWidth
            fullHeight
          >
            <Layout
              fullWidth
              fullHeight
              alignItems={AlignItems.Center}
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
            >
              {renderedImages.length === 0 ? noImagesPlaceholder : renderedImages}
            </Layout>
          </StyledLayout>
          <Layout className="image-preview-modal-footer" fullWidth flexGrow={0} flexShrink={0}>
            <StyledLayout margin={{ x: 4 }} padding={{ y: 1 }} borderTop>
              {"Add multiple images by pasting a file or URL to the below field:"}
              <ImageFriendlyInputForm
                onPasteImage={this.props.onPaste}
                autoFocus={this.props.images && this.props.images.length === 0}
              />
            </StyledLayout>
            <StyledLayout
              display={Display.Flex}
              justifyContent={JustifyContent.Between}
              alignItems={AlignItems.Center}
              fullWidth
              background={Background.Alt2}
              padding={1}
              attachBottom
              position={Position.Absolute}
            >
              <CoreText type={TextType.H4} color={Color.Alt2}>
                {`Adding ${renderedImages.length} ${renderedImages.length > 1 ? "images" : "image"}.`}
              </CoreText>
              <Layout display={Display.Flex}>
                <Layout padding={{ right: 0.5 }}>
                  <Button
                    data-track-click="image-preview-modal-cancel"
                    type={ButtonType.Hollow}
                    onClick={this.props.onCancel}
                  >
                    {"Cancel"}
                  </Button>
                </Layout>
                <Button data-track-click="image-preview-modal-upload" onClick={this.props.onStartUpload}>
                  {"Upload"}
                </Button>
              </Layout>
            </StyledLayout>
          </Layout>
        </StyledLayout>
      </StackedModal>
    );
  }

  private onKeyDown = (e: React.KeyboardEvent<HTMLTextAreaElement>) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      this.props.onStartUpload();
    }
  };
}
