import { action, computed, observable } from "mobx";

import { APIResponse, uploadImage } from "aegis/features/images/rest-uploader";
import { ImageUpload } from "./models";

export type ImageStoreParams = { images: ImageUpload[] };

export class ImageStore {
  @observable private _images: ImageUpload[] = [];
  @observable private _targetUserID: string;

  @computed
  get images(): ImageUpload[] {
    return this._images;
  }

  indexByData(imageDataURLIn: string): number {
    return this._images.findIndex(({ imageDataURL, imageURL }) => {
      return imageDataURL === imageDataURLIn && !imageURL;
    });
  }

  @computed
  get newImages(): ImageUpload[] {
    return this._images.filter(({ imageURL }) => {
      return imageURL == null;
    });
  }

  @computed
  get uploadedImages(): ImageUpload[] {
    return this._images.filter(({ imageURL }) => {
      return !(imageURL == null);
    });
  }

  @computed
  get doneUploading(): boolean {
    return this.newImages.length === 0;
  }

  @computed
  get hasUserID(): boolean {
    return !!this._targetUserID;
  }

  @action
  setTargetUserID(targetID: string) {
    this._targetUserID = targetID;
  }

  @action
  addUploadingImages(images: ImageUpload[]) {
    this._images = this._images.concat(images);
  }

  @action
  updateImage(imageToUpdateData: string, imageUpdateData: Partial<ImageUpload>): ImageUpload | undefined {
    const index = this.indexByData(imageToUpdateData);
    if (index === -1) {
      console.error("Unable to update this image");
      return;
    }
    this._images[index] = { ...this._images[index], ...imageUpdateData };
    return this._images[index];
  }

  @action
  imageUploaded(imageDataURLIn: string, imageURLIn: string) {
    this.updateImage(imageDataURLIn, { imageURL: imageURLIn, error: undefined, uploading: false });
  }

  @action
  imageUploadFailed(imageDataURLIn: string, error: string) {
    this.updateImage(imageDataURLIn, { error: error, uploading: false });
  }

  @action
  startUpload(onSuccessCallback?: (imageURLComment: string) => void) {
    const imagesToUpload = this.newImages;
    if (this._targetUserID) {
      imagesToUpload.forEach(image => {
        this.doImageUpload(image, onSuccessCallback);
      });
    } else {
      console.error("Can't upload images without a target user ID");
    }
  }

  @action
  doImageUpload(image: ImageUpload, onSuccessCallback?: (imageURLComment: string) => void) {
    this.updateImage(image.imageDataURL, { error: undefined, uploading: true });
    const onUploadComplete = ({ data, error }: APIResponse) => {
      if (data) {
        this.imageUploaded(image.imageDataURL, data);
        if (onSuccessCallback) {
          onSuccessCallback(this.formatImageComment(image, data));
        }
      }
      if (error) {
        this.imageUploadFailed(image.imageDataURL, error);
      }
    };
    uploadImage(image.imageFile, onUploadComplete, this._targetUserID);
  }

  formatImageComment = (image: ImageUpload, imageURL: string) => {
    if (image.imageComment) {
      return image.imageComment + "\n" + imageURL;
    }
    return imageURL;
  };

  @action
  retryUpload(imageDataURLIn: string, onSuccessCallback?: (imageURLComment: string) => void) {
    const image = this.updateImage(imageDataURLIn, { error: undefined, imageURL: undefined });
    this.doImageUpload(image!, onSuccessCallback);
  }
}
