import * as React from "react";

import { AutoCompleteInput, Suggestion, SuggestionDisplay } from "aegis/functionality/components/auto-suggest-input";
import { Locale } from "aegis/models";
import {
  Background,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  Overflow,
  StyledLayout,
  TextTransform,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export interface DataProps {
  locales: Locale[];
}

export interface PublicProps {
  onSelect: (code: string | null) => void;
}

type Props = DataProps & PublicProps;

export const LocalesPickerComponent: React.SFC<Props> = ({ locales, onSelect }) => {
  const suggestions: Suggestion[] = locales.map(l => ({
    value: l.code,
    display: (
      <Layout key={l.code} padding={{ left: 0.5 }}>
        <CoreText key={l.code + "-display"} type={TextType.Span}>
          {l.display}
        </CoreText>{" "}
        <CoreText key={l.code + "-code"} type={TextType.Span} transform={TextTransform.Uppercase}>
          ({l.code})
        </CoreText>
      </Layout>
    ),
    match: (input: string) => {
      const inputLowerCase = input.toLowerCase();
      return l.code.toLowerCase().includes(inputLowerCase) || l.display.toLowerCase().includes(inputLowerCase);
    }
  }));

  const emptyOption: Suggestion = {
    value: null,
    display: (
      <Layout key={"all"} padding={{ left: 0.5 }}>
        <CoreText key={"all" + "-display"} type={TextType.Span}>
          {"All"}
        </CoreText>
      </Layout>
    ),
    match: (input: string) => {
      const inputLowerCase = input.toLowerCase();
      return "all".includes(inputLowerCase) || "all".includes(inputLowerCase);
    }
  };

  suggestions.unshift(emptyOption);

  return (
    <StyledLayout
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      background={Background.Base}
      overflow={Overflow.Hidden}
      className="locale-lookup-input"
    >
      <Layout padding={{ y: 0.5 }}>
        <CoreText noWrap>Find other languages: </CoreText>
      </Layout>

      <AutoCompleteInput
        strictMatch
        show={true}
        display={SuggestionDisplay.ALWAYS}
        onSelect={onSelect}
        suggestions={suggestions}
        suggestionCardClassName="locale-lookup-input__suggestions"
      />
    </StyledLayout>
  );
};
