import * as React from "react";

import { inject, observer } from "mobx-react";
import * as FontAwesome from "react-fontawesome";

import { AuthStoreProps, SettingsStoreProps, Theme } from "aegis/stores";
import { AlignItems, Background, Display, Interactable, Layout, StyledLayout, SVG, SVGAsset } from "twitch-core-ui";

type Props = AuthStoreProps & SettingsStoreProps;

@inject("authStore")
@inject("settingsStore")
@observer
export class MeDropdown extends React.Component<Props> {
  public render() {
    const children = [
      this.renderThemeButton(),
      <Interactable data-track-click="me-logout" key="logout" onClick={this.onLogout}>
        <StyledLayout display={Display.Flex} padding={{ x: 2, y: 0.5 }}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ right: 1 }}>
            <SVG asset={SVGAsset.NavLogout} height={18} width={18} />
          </Layout>
          Logout
        </StyledLayout>
      </Interactable>
    ];

    return (
      <StyledLayout padding={{ y: 1 }} background={Background.Alt}>
        {children}
      </StyledLayout>
    );
  }

  private onLogout = () => {
    this.props.authStore!.reset();
  };

  private renderThemeButton = () => {
    const settings = this.props.settingsStore;
    let theme: Theme | null = null;
    let iconName = "";

    // Theme Transition Light -> Dark, Dark -> Pink, Pink -> Light
    switch (settings!.theme) {
      case Theme.Light:
        theme = Theme.Dark;
        iconName = "moon-o";
        break;
      case Theme.Dark:
        theme = Theme.Light;
        iconName = "sun-o";
        break;
      default:
        // Default to light mode and print out a error to console
        theme = Theme.Light;
        iconName = "sun-o";
        console.error("Unrecognized theme");
    }

    return (
      <Interactable
        data-track-click="me-change-theme"
        key="theme"
        onClick={() => this.props.settingsStore!.setTheme(theme!)}
      >
        <StyledLayout display={Display.Flex} padding={{ x: 2, y: 0.5 }}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ right: 1 }}>
            <FontAwesome fixedWidth width={18} name={iconName} />
          </Layout>
          Change Color Theme
        </StyledLayout>
      </Interactable>
    );
  };
}
