import * as React from "react";

import { TopNavDropdown } from "aegis/features/topnav/components/dropdown-menu";
import * as Fragments from "aegis/fragments";
import { Me as MeModel } from "aegis/models";
import { AuthStoreProps } from "aegis/stores";
import { ApolloError } from "apollo-client";
import gql from "graphql-tag";
import { inject } from "mobx-react";

import { Query } from "aegis/functionality/apollo-wrapper";
import { AlignItems, Color, CoreText, Display, Layout, LoadingSpinner, Position, TextType } from "twitch-core-ui";
import { MeComponent, MeProps } from "./component";
import { MeDropdown } from "./components/me-dropdown";

export const FETCH_ME = gql`
  query me {
    me {
      id
      fullName
      admin {
        ...AdminFields
      }
    }
  }
  ${Fragments.AdminFields}
`;

class MeQuery extends Query<MeProps> {}

type Props = AuthStoreProps;

@inject("authStore")
export class Me extends React.Component<Props> {
  public render() {
    return (
      <Layout
        position={Position.Relative}
        display={Display.Flex}
        flexGrow={1}
        alignItems={AlignItems.Stretch}
        fullHeight
      >
        <MeQuery
          displayName="Me"
          query={FETCH_ME}
          fetchPolicy={"network-only"}
          onCompleted={data => {
            if (data) {
              const meData = data as MeProps;

              if (meData && meData.me && meData.me.admin) {
                this.props.authStore!.userLogin(meData.me.admin);
              }
            }
          }}
        >
          {({ loading, error, data }) => {
            return (
              <TopNavDropdown
                topnavElement={this.renderClickableCard(data && data.me, loading, error)}
                dropdownElement={<MeDropdown />}
                display={Display.Flex}
                fullWidth
              />
            );
          }}
        </MeQuery>
      </Layout>
    );
  }

  private renderClickableCard(me: MeModel | undefined, loading: boolean, error: ApolloError | undefined) {
    if (loading) {
      return <LoadingSpinner inheritColor />;
    }
    if (error) {
      return (
        <CoreText type={TextType.Strong} color={Color.Error}>
          {error.message}
        </CoreText>
      );
    }
    if (!me) {
      return <div />;
    }

    return <MeComponent me={me} />;
  }
}
