import * as React from "react";

import * as Enzyme from "enzyme";
import { mount, shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { BanFormModal } from "aegis/features/banform";
import { mockedBanFormQueries } from "aegis/features/banform/mocks";
import { STRIKE_BUTTON_TEST_SELECTOR } from "aegis/features/strike-button";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { MessageLog, Props } from "./component";
import { getCellClassNames, MessageLogRow } from "./components/message-log-row";
import { fakeMessage } from "./mocks";

import "aegis/functionality/date-extensions";
import { DEFAULT_CHAT_COLUMNS, MessageColumn } from "./models";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<Props>, shouldMount?: boolean) => {
  const combinedProps = {
    entries: [fakeMessage, fakeMessage, fakeMessage, fakeMessage],
    chatColumns: DEFAULT_CHAT_COLUMNS,
    ...overrides
  };
  let wrapper;
  if (shouldMount) {
    wrapper = mount(wrapWithGQLProvider(<MessageLog {...combinedProps} />, mockedBanFormQueries).wrappedComponent);
  } else {
    wrapper = shallow(<MessageLog {...combinedProps} />);
  }
  return {
    props: combinedProps,
    wrapper
  };
};

describe("Chat log", () => {
  it("Renders all entries as ChatLogRows with a small amount of chat entries", () => {
    const { wrapper } = setup();
    expect(wrapper.find(MessageLogRow).length).toBe(4);
  });

  it("Renders all entries as ChatLogRows with a large amount of chat entries", () => {
    const entries = [];
    for (var i = 0; i < 4000; i++) {
      entries.push(fakeMessage);
    }
    const { wrapper } = setup({ entries });
    expect(wrapper.find(MessageLogRow).length).toBe(4000);
  });

  it("Renders with an empty list of chat entries", () => {
    const { wrapper } = setup({ entries: [] });
    expect(wrapper.find(MessageLogRow).length).toBe(0);
  });

  describe("Column Header Ordering", () => {
    it("falls back to default ordering if no prop", () => {
      const { wrapper } = setup();
      expect(wrapper).toMatchSnapshot();
    });

    it("Renders the column headers based on chatColumns prop", () => {
      const reverseOrder = DEFAULT_CHAT_COLUMNS.slice().reverse() as MessageColumn[];
      const { wrapper } = setup({ chatColumns: reverseOrder });
      expect(wrapper).toMatchSnapshot();

      const columnHeaders = wrapper.find({ className: "messagerow" });
      expect(columnHeaders.children()).toHaveLength(reverseOrder.length);

      DEFAULT_CHAT_COLUMNS.forEach((column, index) => {
        expect(wrapper.find({ className: getCellClassNames(column) })).toEqual(
          columnHeaders.childAt(reverseOrder.length - 1 - index)
        );
      });
    });
  });

  describe("Chat log strike form", () => {
    const modalRoot = document.createElement("div");
    modalRoot.setAttribute("id", "portal-modal-root");
    const body = document.querySelector("body");
    if (body) {
      body.appendChild(modalRoot);
    }

    it("renders the Strike Form when strike button is clicked in row", () => {
      // Suppress query error (I'm not writing intricate mocks for BanForm to test the strike button)
      spyOn(console, "error");

      const { wrapper } = setup({ entries: [fakeMessage] }, true);
      const button = wrapper.find({ "data-test-selector": STRIKE_BUTTON_TEST_SELECTOR }).first();
      expect(wrapper.find(BanFormModal)).toHaveLength(0);
      button.simulate("click");
      expect(wrapper.find(BanFormModal)).toHaveLength(1);
      expect(true).toBeTruthy();
    });
  });
});
