import * as classNames from "classnames";
import * as React from "react";

import { copyTextToClipboard } from "aegis/functionality/utils/contrib/twilight/copy-text-to-clipboard";
import { getKeysOfSelection } from "aegis/functionality/utils/get-keys";
import { Background, CoreText, InjectStyledLayout, StyledLayout, TextType } from "twitch-core-ui";
import { getCellClassNames, HighlightType, MessageLogRow } from "./components/message-log-row";
import { DEFAULT_CHAT_COLUMNS, formatMessageEntry, Message, MessageColumn } from "./models";

import "./style.scss";

export type HighlightedRows = { [key: string]: HighlightType };

export interface Props {
  className?: string;
  entries: Message[];
  columnNameOverride?: Map<MessageColumn, string>;
  additionalActionFactory?: (message: Message) => JSX.Element[];
  highlights?: HighlightedRows; // IDs of the rows to high light
  chatColumns?: MessageColumn[];
}

export enum TestSelectors {
  MessageRow = "message-log-message-row"
}

export class MessageLog extends React.Component<Props, {}> {
  private messageBodyRef: HTMLElement | null = null;
  public render() {
    const { entries, highlights, additionalActionFactory, chatColumns, className } = this.props;

    const columnsToRender = chatColumns || DEFAULT_CHAT_COLUMNS;
    const messageLogRows: JSX.Element[] = [];
    const hiddenMessageBodies: JSX.Element[] = [];
    entries.forEach((entry, index) => {
      let additionalActions: JSX.Element[] = [];
      if (additionalActionFactory) {
        additionalActions = additionalActionFactory(entry);
      }

      messageLogRows.push(
        <MessageLogRow
          entry={entry}
          key={entry.id}
          index={index}
          highlight={highlights && highlights[entry.id]}
          additionalActions={additionalActions}
          data-test-selector={TestSelectors.MessageRow}
          chatColumns={columnsToRender}
        />
      );
      hiddenMessageBodies.push(
        <div className="message-body" data-key={index} key={entry.id}>
          {entry.body}
        </div>
      );
    });
    return (
      <div className={classNames(`message-grid-${columnsToRender.length}`, className)} onCopy={this.onCopy}>
        <StyledLayout className="messagerow">
          {columnsToRender.map((column: MessageColumn) => {
            return (
              <InjectStyledLayout
                key={column}
                border
                background={Background.Alt}
                fullWidth={column === MessageColumn.Message}
                className={getCellClassNames(MessageColumn[column])}
              >
                <CoreText bold type={TextType.Span}>
                  {this.getColumnName(column)}
                </CoreText>
              </InjectStyledLayout>
            );
          })}
        </StyledLayout>
        {messageLogRows}
        <div className="hidden" ref={ref => (this.messageBodyRef = ref)}>
          {hiddenMessageBodies}
        </div>
      </div>
    );
  }

  private getColumnName = (column: MessageColumn) => {
    const { columnNameOverride } = this.props;
    if (columnNameOverride && columnNameOverride.get(column)) {
      return columnNameOverride.get(column);
    }
    return MessageColumn[column];
  };

  private onCopy = (event: React.ClipboardEvent<HTMLElement>) => {
    const selection = window.getSelection();
    const { first, last } = getKeysOfSelection(selection, "messagerow");

    if (first === last) {
      return;
    }

    // Prevent race condition of initial copy propogating after this function
    event.preventDefault();

    const selectedEntries = this.props.entries.slice(first, last ? last : this.props.entries.length);

    const messageBodies = this.messageBodyRef!.childNodes;

    // Stores selected range before it is wiped by copyTextToClipboard
    const selectedRange = selection.getRangeAt(0);
    copyTextToClipboard(
      selectedEntries
        .map((entry, index) => {
          const body = messageBodies.item(first + index).textContent;
          return formatMessageEntry(entry, body!);
        })
        .join("\n")
    );

    // Restores selected area, unselected by copyTextToClipboard
    selection.removeAllRanges();
    selection.addRange(selectedRange);
  };
}
