import * as React from "react";

import * as Enzyme from "enzyme";
import { mount, shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { mockedBanFormQueries } from "aegis/features/banform/mocks";
import { Props } from "aegis/features/messages";
import { fakeMessage, ResponseWithNTwitchWhisperMessages } from "aegis/features/messages/mocks";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { CollapsibleMessageLog, TestSelectors } from "./component";

import "aegis/functionality/date-extensions";
import { HighlightedRows } from "../../component";
import { HighlightType } from "../message-log-row";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<Props>, shouldMount?: boolean) => {
  const combinedProps = {
    entries: [fakeMessage, fakeMessage, fakeMessage, fakeMessage],
    ...overrides
  };
  let wrapper;
  if (shouldMount) {
    wrapper = mount(
      wrapWithGQLProvider(<CollapsibleMessageLog {...combinedProps} />, mockedBanFormQueries).wrappedComponent
    );
  } else {
    wrapper = shallow(<CollapsibleMessageLog {...combinedProps} />);
  }
  return {
    props: combinedProps,
    wrapper
  };
};

describe("Collapsible Message Log", () => {
  it("shows more entries when the 'Show More' button is clicked", () => {
    const entries = ResponseWithNTwitchWhisperMessages(10);
    const { wrapper } = setup({ entries }, true);

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(5);

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.ShowMoreButton })
        .hostNodes()
        .text()
    ).toBe("Show More");

    wrapper
      .find({ "data-test-selector": TestSelectors.ShowMoreButton })
      .hostNodes()
      .simulate("click");

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(10);
  });

  it("does not render the showMore button when entries length is greater than 0 and less than or equal to 5", () => {
    const entries = ResponseWithNTwitchWhisperMessages(3);
    const { wrapper } = setup({ entries });

    expect(wrapper.find({ "data-test-selector": TestSelectors.ShowMoreButton }).hostNodes().length).toBe(0);

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).length).toBe(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(3);
  });

  it("shows less entries when the 'Show Less' button is clicked", () => {
    const entries = ResponseWithNTwitchWhisperMessages(10);
    const { wrapper } = setup({ entries }, true);

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(5);

    wrapper
      .find({ "data-test-selector": TestSelectors.ShowMoreButton })
      .hostNodes()
      .simulate("click");

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.ShowMoreButton })
        .hostNodes()
        .text()
    ).toBe("Show Less");

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(10);

    wrapper
      .find({ "data-test-selector": TestSelectors.ShowMoreButton })
      .hostNodes()
      .simulate("click");

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(5);
  });

  it("always shows highlighted messages", () => {
    const entries = ResponseWithNTwitchWhisperMessages(3);
    const highlights: HighlightedRows = {};
    highlights[entries[0].id] = HighlightType.Main;

    const { wrapper } = setup({ entries, highlights });

    expect(wrapper.find({ "data-test-selector": TestSelectors.ShowMoreButton }).hostNodes().length).toBe(0);

    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).length).toBe(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.MessageEntries }).prop("entries").length).toBe(3);
    expect(wrapper.find({ "data-test-selector": TestSelectors.HighlightedEntries })).toBeTruthy();
  });
});
