import * as React from "react";

import { shallow } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { Message } from "../..";
import { EmoteText } from "../../emote-text";
import { fakeMessage } from "../../mocks";
import { DEFAULT_CHAT_COLUMNS, MessageColumn } from "../../models";
import { baseClassName, getCellClassNames, HighlightType, MessageLogRow, TestSelectors } from "./component";

import { StyledLayoutProps } from "twitch-core-ui";

import "aegis/functionality/date-extensions.ts";

Enzyme.configure({ adapter: new Adapter() });

describe("Message Log Row", () => {
  it("renders mesage log row", () => {
    const props = { entry: fakeMessage, chatColumns: DEFAULT_CHAT_COLUMNS };
    const wrapper = shallow(<MessageLogRow {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.SystemMessage }).length).toBe(0);
    expect(wrapper.find(EmoteText).length).toBe(1);
  });

  it("renders mesage log row with no highlight", () => {
    const props = { entry: fakeMessage, chatColumns: DEFAULT_CHAT_COLUMNS, highlight: undefined };
    const wrapper = shallow(<MessageLogRow {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Wrapper }).prop("className")).not.toContain("highlight");
  });

  it("renders mesage log row with an specific highlight", () => {
    const props = { entry: fakeMessage, chatColumns: DEFAULT_CHAT_COLUMNS, highlight: HighlightType.Main };
    const wrapper = shallow(<MessageLogRow {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.Wrapper }).prop("className")).toContain(
      `highlight-${HighlightType.Main}`
    );
  });

  it("renders system message if available", () => {
    const userNoticeEntry = { ...fakeMessage, type: "usernotice", systemMessage: "All systems are go" };
    const props = { entry: userNoticeEntry, chatColumns: DEFAULT_CHAT_COLUMNS };
    const wrapper = shallow(<MessageLogRow {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.SystemMessage }).length).toBe(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SystemMessage }).html()).toContain(
      userNoticeEntry.systemMessage
    );
  });

  it("renders with additional actions if provided", () => {
    const props = { entry: fakeMessage, chatColumns: DEFAULT_CHAT_COLUMNS };
    const testSelector = "additional-action";
    const additionalAction = <div key={fakeMessage.id} data-test-selector={testSelector} />;
    const wrapper = shallow(<MessageLogRow additionalActions={[additionalAction]} {...props} />);

    expect(wrapper.find({ "data-test-selector": testSelector }).length).toBe(1);
  });

  it("does not render target user if target does not exist", () => {
    const noTargetEntry: Message = { ...fakeMessage, target: undefined };
    const props = { entry: noTargetEntry, chatColumns: DEFAULT_CHAT_COLUMNS };
    const wrapper = shallow(<MessageLogRow {...props} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.SystemMessage }).length).toBe(0);
    expect(wrapper.find(EmoteText).length).toBe(1);
  });

  it("renders when deleted", () => {
    const deletedEntry: Message = { ...fakeMessage, isDeleted: true };
    const props = { entry: deletedEntry, chatColumns: DEFAULT_CHAT_COLUMNS };
    const wrapper = shallow(<MessageLogRow {...props} />);

    const styleProps = wrapper
      .find({ "data-test-selector": TestSelectors.MessageBodyColumn })
      .props() as StyledLayoutProps;
    expect(styleProps.className).toContain(`${baseClassName}deleted`);
    expect(wrapper.find(EmoteText).length).toBe(1);
  });

  it("renders cells based column order", () => {
    const reverseOrder = DEFAULT_CHAT_COLUMNS.slice().reverse() as MessageColumn[];
    const props = { entry: fakeMessage, chatColumns: reverseOrder };
    const wrapper = shallow(<MessageLogRow {...props} />);

    DEFAULT_CHAT_COLUMNS.forEach((column, index) => {
      if (column !== MessageColumn.Message) {
        expect(wrapper.find({ className: getCellClassNames(column) })).toEqual(
          wrapper.find({ className: "messagerow" }).childAt(reverseOrder.length - 1 - index)
        );
      } else {
        expect(wrapper.find({ "data-test-selector": TestSelectors.MessageBodyColumn }).parent()).toEqual(
          wrapper.find({ className: "messagerow" }).childAt(reverseOrder.length - 1 - index)
        );
      }
    });
  });
});
