import * as React from "react";

import { StrikeButton } from "aegis/features/strike-button";
import { TranslateButton } from "aegis/features/translate";
import { User } from "aegis/features/user";
import { errorNotification } from "aegis/functionality/error-notifications";
import { hasClassInAncestry } from "aegis/functionality/utils/get-keys";
import { formatMessageEntry, Message, MessageColumn } from "../..";
import { EmoteText } from "../../emote-text";

import {
  AlignItems,
  Background,
  ButtonType,
  Color,
  CoreText,
  Display,
  StyledLayout,
  StyledLayoutProps,
  TextTransform,
  TextType,
  Tooltip,
  WordBreak
} from "twitch-core-ui";

import "./style.scss";

export enum HighlightType {
  Alt = "alt",
  Main = "main"
}

export interface Props {
  entry: Message;
  index?: Number;
  highlight?: HighlightType;
  additionalActions?: JSX.Element[]; // If specified, additional actions will be rendered after actions in the same column
  chatColumns: MessageColumn[];
}

export enum TestSelectors {
  SystemMessage = "system-message",
  MessageBodyColumn = "message-row--message-body-column",
  Wrapper = "wrapper"
}
export const baseClassName = "aegis-messagelog__";
export const getCellClassNames = (type: String) => {
  return `${baseClassName}cell ${baseClassName}${type.toLowerCase()}`;
};

// Hijack double click function, used to force selection to single cell
// (Without this, Chrome auto-selects the next "cell" with selectable text)
const focusOnClick = (event: React.MouseEvent) => {
  if (event.target instanceof Element && hasClassInAncestry(event.target, "combo-breaker")) {
    return;
  }
  event.preventDefault();
  if (event.target instanceof HTMLElement) {
    const selection = window.getSelection();
    const range = document.createRange();
    range.selectNodeContents(event.target);
    selection.removeAllRanges();
    selection.addRange(range);
  }
  return;
};

export const MessageLogRow: React.SFC<Props> = props => {
  const { entry, index, additionalActions, chatColumns, highlight } = props;

  const { type, body, systemMessage, isDeleted } = entry;

  const displayedText = systemMessage ? `${systemMessage}\n${body}` : body;

  const ts = new Date(parseInt(entry.sentTimestamp, 10));

  const className = !highlight ? "messagerow" : `messagerow highlight-${highlight}`;

  const verticalCenterProps: Partial<StyledLayoutProps> = {
    display: Display.Flex,
    alignItems: AlignItems.Center
  };

  const textColor = type === "clearchat" ? Color.Error : Color.Base;
  return (
    <div
      className={className}
      key={entry.id}
      data-key={index}
      onDoubleClick={focusOnClick}
      data-test-selector={TestSelectors.Wrapper}
    >
      {chatColumns.map((column: MessageColumn) => {
        const key = `${entry.id}-${column}`;
        switch (column) {
          case MessageColumn.Time:
            return (
              <StyledLayout
                className={getCellClassNames(MessageColumn[MessageColumn.Time])}
                key={key}
                {...verticalCenterProps}
              >
                <CoreText type={TextType.Span} noWrap>
                  {ts.toDisplayFormat()}
                </CoreText>
              </StyledLayout>
            );
          case MessageColumn.Type:
            return (
              <StyledLayout
                className={getCellClassNames(MessageColumn[MessageColumn.Type])}
                key={key}
                {...verticalCenterProps}
              >
                <CoreText transform={TextTransform.Uppercase} type={TextType.Span} noWrap>
                  {type}
                </CoreText>
              </StyledLayout>
            );
          case MessageColumn.Target:
            return (
              <StyledLayout
                className={getCellClassNames(MessageColumn[MessageColumn.Target])}
                key={key}
                {...verticalCenterProps}
              >
                {entry.target && <User user={entry.target} showStatus showType inline />}
              </StyledLayout>
            );
          case MessageColumn.From:
            return (
              <StyledLayout
                className={getCellClassNames(MessageColumn[MessageColumn.From])}
                key={key}
                {...verticalCenterProps}
              >
                {entry.from && <User user={entry.from} badges={entry.badges} inline />}
              </StyledLayout>
            );
          case MessageColumn.Message:
            return (
              <StyledLayout className={`${baseClassName}cell`} key={key} {...verticalCenterProps}>
                <StyledLayout
                  className={`${baseClassName}message${isDeleted ? ` ${baseClassName}deleted` : ""}`}
                  data-test-selector={TestSelectors.MessageBodyColumn}
                >
                  {systemMessage && (
                    <Tooltip label="System Message" display={Display.Block}>
                      <StyledLayout padding={{ left: 0.5 }} background={Background.Alt2} color={Color.Link}>
                        <CoreText
                          type={TextType.P}
                          color={Color.Alt2}
                          bold={true}
                          noWrap={false}
                          wordBreak={WordBreak.BreakWord}
                          data-test-selector={TestSelectors.SystemMessage}
                        >
                          {systemMessage}
                        </CoreText>
                      </StyledLayout>
                    </Tooltip>
                  )}
                  <CoreText type={TextType.Span} color={textColor} noWrap={false} wordBreak={WordBreak.BreakWord}>
                    <EmoteText text={body} emotes={entry.emotes} />
                  </CoreText>
                </StyledLayout>
              </StyledLayout>
            );
          case MessageColumn.Actions:
            return (
              <StyledLayout
                className={getCellClassNames(MessageColumn[MessageColumn.Actions])}
                key={key}
                {...verticalCenterProps}
              >
                <TranslateButton for={displayedText} type={ButtonType.Hollow} />
                <StrikeButton
                  banFormProps={{
                    targetUserID: entry.from && entry.from.id,
                    targetUser: entry.from && entry.from.login,
                    defaultContent: "chat",
                    placeholders: { chatlog: formatMessageEntry(entry) }
                  }}
                  user={entry.from}
                  type={ButtonType.Hollow}
                />
                {additionalActions}
              </StyledLayout>
            );
          default:
            errorNotification(`Unrecognized message column type column: ${column}, message-id: ${entry.id}`, {
              error: `MessageLogRow unrecognized message column type, column: ${column}, message-id: ${entry.id}`
            });
            return (
              <CoreText type={TextType.Span} color={Color.Error} key={`${column}-${entry.id}-error`}>
                ERROR
              </CoreText>
            );
        }
      })}
    </div>
  );
};
