import * as React from "react";

import { Emote } from "aegis/features/emote";
import { MessageEmote } from "../models";

export interface PublicProps {
  text: string;
  emotes: MessageEmote[];
}

// Used internally to keep track of emote positions in the array
export const emoteReplacementPuaChar = "\uE010";

export const EmoteText: React.StatelessComponent<PublicProps> = ({ text, emotes }) => {
  // If there's no emote in the text, use the original text
  if (!emotes || emotes.length === 0) {
    return <span key="text-1">{text}</span>;
  }

  let isAction = false;
  const actionMatch = new RegExp("^\u0001ACTION (.+)\u0001$");
  if (actionMatch.test(text)) {
    isAction = true;
    text = text.replace(actionMatch, function(_: {}, msg: string): string {
      return msg;
    });
  }

  const chars = Array.from(text.replace(emoteReplacementPuaChar, ""));
  const images: { [key: number]: JSX.Element } = {};

  emotes.forEach(emote => {
    const emoteCode = chars.slice(emote.start, emote.end + 1).join("");
    images[emote.start] = <Emote key={`emote-${emote.start}`} id={emote.emote.id} emoteCode={emoteCode} version={1} />;
    for (let j = emote.start; j <= emote.end; j += 1) {
      chars[j] = emoteReplacementPuaChar;
    }
  });

  const elements: JSX.Element[] = [];
  let segmentStart = 0;
  let isEmoteSegment = false;
  for (let index = 0; index < chars.length; index += 1) {
    if (chars[index] === emoteReplacementPuaChar) {
      if (!isEmoteSegment) {
        if (index !== 0) {
          elements.push(<span key={`text-${segmentStart}`}>{chars.slice(segmentStart, index).join("")}</span>);
        }
        isEmoteSegment = true;
        segmentStart = index;
      }
      if (images.hasOwnProperty(index)) {
        elements.push(images[index]);
      }
    } else {
      if (isEmoteSegment) {
        isEmoteSegment = false;
        segmentStart = index;
      }
    }
  }

  // If we didn't end in an emote, emit the rest of the line
  if (!isEmoteSegment) {
    elements.push(<span key={`text-${segmentStart}`}>{chars.slice(segmentStart, chars.length).join("")}</span>);
  }

  return isAction ? (
    <span>
      {"\u0001"}ACTION {elements}
      {"\u0001"}
    </span>
  ) : (
    <span>{elements}</span>
  );
};
