import { Badge, getNameAsDisplayed, User } from "aegis/models";

export type Message = {
  id: string;
  sentTimestamp: string;
  type: string;
  from?: User;
  target?: User;

  emotes: MessageEmote[];
  badges: Badge[];
  body: string;
  isDeleted: boolean;
  systemMessage?: string;
};

export type Emote = {
  id: string;
};

export type MessageEmote = {
  emote: Emote;
  start: number;
  end: number;
};

export function formatMessageEntry(entry: Message, translatedBody?: string): string {
  const ts = new Date(parseInt(entry.sentTimestamp, 10));
  const { from, target, body, systemMessage, type, badges } = entry;
  let badgeText = " ";
  if (badges) {
    badges.forEach(badge => {
      badgeText = badgeText + badge.title + " ";
    });
  }
  const messageBody = translatedBody && translatedBody !== body ? `${body} (Translation: ${translatedBody})` : body;

  const messageTypeTag = type === "clearchat" ? " ACTION" : "";
  const userMessage = `[${ts.toDisplayFormat()}][#${target && target.displayName} (${target &&
    target.id})] [${badgeText}] ${from && getNameAsDisplayed(from)} (${from &&
    from.id})${messageTypeTag}: ${messageBody}`;
  const sysMessage = `System Message: ${systemMessage}`;

  const formattedMessage = systemMessage ? `${sysMessage}\n${userMessage}` : userMessage;
  return formattedMessage;
}

export enum MessageColumn {
  Time = "Time",
  Type = "Type",
  From = "From",
  Target = "Target",
  Message = "Message",
  Actions = "Actions"
}

export const ChatColumnNameOverrides = new Map<MessageColumn, string>([
  [MessageColumn.From, "User"],
  [MessageColumn.Target, "Channel"],
  [MessageColumn.Message, "Message / Reason"]
]);

export const DEFAULT_CHAT_COLUMNS: MessageColumn[] = [
  MessageColumn.Time,
  MessageColumn.Type,
  MessageColumn.Target,
  MessageColumn.From,
  MessageColumn.Message,
  MessageColumn.Actions
];
