import * as React from "react";

import { errorNotification } from "aegis/functionality/error-notifications";
import {
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  SegmentedButton,
  SegmentedButtonOption,
  TextArea,
  TextType
} from "twitch-core-ui";
import { UpdatePartnerEscalationListResponse, UpdateType } from "../mutations";

import "./styles.scss";

export enum PartnerActions {
  BulkAdd = "Bulk add/overwrite",
  OneTimeAdd = "One-time add",
  OneTimeRemove = "One-time remove"
}

const PartnerActionToUpdateTypeMap = new Map<PartnerActions, UpdateType>([
  [PartnerActions.OneTimeAdd, UpdateType.Add],
  [PartnerActions.OneTimeRemove, UpdateType.Remove],
  [PartnerActions.BulkAdd, UpdateType.Replace]
]);

export enum TestSelectors {
  ActionButton = "partner-escalation-action-button",
  ResultBefore = "partner-escalation-result-before",
  ResultAfter = "partner-escalation-result-after"
}

export interface Props {
  updateList: (users: string[], type: UpdateType) => void;
  disabled?: boolean;
  result?: UpdatePartnerEscalationListResponse;
}

export const ChangeUsersComponent: React.SFC<Props> = ({ updateList, disabled, result }) => {
  const [action, setAction] = React.useState<PartnerActions>(PartnerActions.BulkAdd);
  const [inputUsers, changeInputUsers] = React.useState("");

  const fireAction = () => {
    const userIdentifiers = inputUsers.trim().split(/[\n,]+/);

    const type = PartnerActionToUpdateTypeMap.get(action);
    if (!type) {
      errorNotification("Unrecognized action: " + action);
      return;
    }
    updateList(userIdentifiers, type);
  };

  return (
    <Layout>
      <Layout>
        <CoreText type={TextType.H4} bold>
          Action
        </CoreText>
        <Layout padding={{ y: 0.5 }}>
          <SegmentedButton>
            <SegmentedButtonOption
              name="partner-actions"
              label={PartnerActions.BulkAdd}
              checked={action === PartnerActions.BulkAdd}
              onChange={() => setAction(PartnerActions.BulkAdd)}
            />
            <SegmentedButtonOption
              name="partner-actions"
              label={PartnerActions.OneTimeAdd}
              checked={action === PartnerActions.OneTimeAdd}
              onChange={() => setAction(PartnerActions.OneTimeAdd)}
            />
            <SegmentedButtonOption
              name="partner-actions"
              label={PartnerActions.OneTimeRemove}
              checked={action === PartnerActions.OneTimeRemove}
              onChange={() => setAction(PartnerActions.OneTimeRemove)}
            />
          </SegmentedButton>
        </Layout>
      </Layout>
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column} padding={{ top: 2 }}>
        <CoreText type={TextType.H4} bold>
          Twitch User IDs
        </CoreText>
        <CoreText>Separate users by line or by comma</CoreText>
        <Layout padding={{ y: 0.5 }} className="partner-escalation-input-textarea">
          <TextArea onChange={e => changeInputUsers(e.currentTarget.value)} />
        </Layout>
        <Layout>
          <Button
            data-test-selector={TestSelectors.ActionButton}
            onClick={fireAction}
            disabled={disabled}
            type={!!result ? ButtonType.Success : undefined}
          >
            {action}
          </Button>
        </Layout>
        {!!result && (
          <Layout>
            <CoreText color={Color.Alt2} data-test-selector={TestSelectors.ResultBefore}>
              {`Total Before: ${result.updatePartnerEscalationList.before}`}
            </CoreText>
            <CoreText color={Color.Alt2} data-test-selector={TestSelectors.ResultAfter}>
              {`Total After: ${result.updatePartnerEscalationList.after}`}
            </CoreText>
          </Layout>
        )}
      </Layout>
    </Layout>
  );
};
