import * as React from "react";

import * as QueryString from "query-string";
import { RouteComponentProps, withRouter } from "react-router";
import { TabItemPublicProps, Tabs, TabSize } from "twitch-core-ui";

import { QueryProps } from "aegis/pages/partner-escalation-list";

export enum EscalationTab {
  ChangeUsers = "ChangeUsers",
  ViewList = "ViewList"
}

export const tabsData = [
  { tab: EscalationTab.ChangeUsers, label: "Add or Remove Users" },
  { tab: EscalationTab.ViewList, label: "View and Manage List" }
];

export function getCurrentTab(tabId?: string): EscalationTab | null {
  if (tabId && tabId in EscalationTab) {
    return tabId as EscalationTab;
  }
  return null;
}

export interface PublicProps {
  queryProps: QueryProps;
}

type RouteProps = RouteComponentProps<{}>;
export type Props = PublicProps & RouteProps;

function getActiveTabIndex(currentTab: EscalationTab) {
  return tabsData.findIndex(({ tab }) => {
    return currentTab === tab;
  });
}

export class PartnerEscalationTabsComponent extends React.PureComponent<Props> {
  public constructor(props: Props) {
    super(props);
    if (!this.currentTab) {
      this.changeTab(EscalationTab.ChangeUsers);
    }
  }

  public render() {
    if (!this.currentTab) {
      return null;
    }
    const tabs: TabItemPublicProps[] = tabsData.map(({ tab, label }) => {
      return {
        label: label,
        onClick: () => {
          this.changeTab(tab);
        },
        "data-test-selector": tab
      };
    });
    return (
      <Tabs
        tabs={tabs}
        size={TabSize.Large}
        activeTabIndex={getActiveTabIndex(this.currentTab)}
        data-track-click="partner-escalation-tabs"
      />
    );
  }

  private get currentTab() {
    return getCurrentTab(this.props.queryProps.tabID);
  }

  private changeTab = (tab: EscalationTab) => {
    const queryParams = QueryString.parse(location.search) as QueryProps;
    const updatedParams = QueryString.stringify({ ...queryParams, tabID: tab });
    this.props.history.push({ pathname: "/partnerescalation", search: updatedParams });
  };
}

export const PartnerEscalationTabs: React.ComponentClass<PublicProps> = withRouter<Props>(
  PartnerEscalationTabsComponent
);
