import * as React from "react";

import * as QueryString from "query-string";
import { RouteComponentProps, withRouter } from "react-router";

import { QueryProps } from "aegis/pages/partner-escalation-list";
import {
  CoreText,
  Display,
  JustifyContent,
  Layout,
  Pagination,
  PaginationMouseEvent,
  StyledLayout,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  TextType
} from "twitch-core-ui";
import { RemoveFromListButton } from "./remove-from-list-button";

import "./styles.scss";

export enum TestSelectors {
  LoginCell = "partner-escalation-list-login-cell",
  RecordCount = "partner-escalation-record-count",
  PartnerTableRow = "partner-escalation-table-row"
}

export interface EscalatedPartners {
  id: string;
  login: string;
  source: string;
  dateAdded: string;
}

type RouteProps = RouteComponentProps<{}>;
export interface PublicProps {
  users: EscalatedPartners[];
  total: number;
  currentPage: number;
  totalPages: number;
}
export type Props = PublicProps & RouteProps;

export class ViewPartnerListComponent extends React.Component<Props> {
  public render() {
    const { users, total, currentPage, totalPages } = this.props;
    return (
      <Layout fullWidth display={Display.Flex} justifyContent={JustifyContent.Center}>
        <Layout className="partner-view-table">
          <Layout fullWidth textAlign={TextAlign.Right} padding={{ bottom: 1 }}>
            <CoreText
              type={TextType.H4}
              data-test-selector={TestSelectors.RecordCount}
            >{`${total} Total Records`}</CoreText>
          </Layout>
          <StyledLayout elevation={1}>
            <Table>
              <TableHeader>
                <TableHeading label="ID" />
                <TableHeading label="Login" />
                <TableHeading label="Source" />
                <TableHeading label="Date Added" />
                <TableHeading label="Remove" />
              </TableHeader>
              <TableBody>
                {users.map(user => {
                  return (
                    <TableRow data-test-selector={TestSelectors.PartnerTableRow} key={user.id}>
                      <TableCell>{user.id}</TableCell>
                      <TableCell data-test-selector={TestSelectors.LoginCell}>{user.login}</TableCell>
                      <TableCell>{user.source}</TableCell>
                      <TableCell>{user.dateAdded}</TableCell>
                      <TableCell>
                        <RemoveFromListButton user={user} />
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          </StyledLayout>
          <Layout padding={{ top: 1 }}>
            <Pagination
              currentPage={currentPage}
              totalPages={totalPages}
              onClickIndex={(e: PaginationMouseEvent<HTMLDivElement>): void => {
                this.changePage(e.goToPage);
              }}
              onClickNext={() => this.changePage(currentPage + 1)}
              onClickPrevious={() => this.changePage(currentPage - 1)}
              nextPageButtonAriaLabel="Next Page"
              previousPageButtonAriaLabel="Previous Page"
            />
          </Layout>
        </Layout>
      </Layout>
    );
  }

  private changePage = (newPage: number) => {
    const queryParams = QueryString.parse(location.search) as QueryProps;
    const updatedParams = QueryString.stringify({ ...queryParams, page: newPage.toString() });
    this.props.history.push({ pathname: "/partnerescalation", search: updatedParams });
  };
}

export const ViewPartnerListComponentWrapper: React.ComponentClass<PublicProps> = withRouter<Props>(
  ViewPartnerListComponent
);
