import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { User } from "aegis/models";

import { Query } from "aegis/functionality/apollo-wrapper";
import { LoadingSpinner } from "twitch-core-ui";
import { EscalatedPartners, ViewPartnerListComponentWrapper } from "./component";

export const PARTNER_ESCALATION_LIMIT = 20;

export const PARTNER_ESCALATIONS_QUERY = gql`
  query escalationPartners($first: Int, $offset: Int) {
    escalationPartners(first: $first, offset: $offset) {
      edges {
        node {
          id
          user {
            ...UserFields
          }
          createdAt
          updatedAt
          override
        }
      }
      pageInfo {
        total
        offset
      }
    }
  }
  ${Fragments.UserFields}
`;

type Variables = {
  first: number;
  offset: number;
};

interface EscalationPartnerEdge {
  node: {
    id: string;
    user: User;
    createdAt: string;
    updatedAt: string;
    override: string;
  };
}

interface Data {
  escalationPartners?: {
    edges: EscalationPartnerEdge[];
    pageInfo: {
      total: number;
      offset: number;
    };
  };
}

class PartnerListQuery extends Query<Data, Variables> {}

export interface PublicProps {
  pageNumber?: number;
}

export const ViewPartnerList: React.SFC<PublicProps> = ({ pageNumber }) => {
  return (
    <PartnerListQuery
      displayName="ViewPartnerList"
      query={PARTNER_ESCALATIONS_QUERY}
      variables={{ first: PARTNER_ESCALATION_LIMIT, offset: pageNumber ? pageNumber * PARTNER_ESCALATION_LIMIT : 0 }}
    >
      {({ loading, error, data }) => {
        if (loading) {
          return <LoadingSpinner />;
        }
        if (error || !data || !data.escalationPartners) {
          return <div>Failed to load escalation partners</div>;
        }
        if (data && data.escalationPartners) {
          const formattedData: EscalatedPartners[] = data.escalationPartners.edges.map(({ node }) => {
            return {
              id: node.user.id,
              login: node.user.login,
              source: node.override,
              dateAdded: node.createdAt
            };
          });
          return (
            <ViewPartnerListComponentWrapper
              users={formattedData}
              total={data.escalationPartners.pageInfo.total}
              currentPage={pageNumber ? pageNumber + 1 : 1}
              totalPages={Math.ceil(data.escalationPartners.pageInfo.total / PARTNER_ESCALATION_LIMIT)}
            />
          );
        }
      }}
    </PartnerListQuery>
  );
};
