import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { ClickOutDetector } from "aegis/functionality/components/click-out-detector";
import { DropdownMenu } from "aegis/functionality/components/dropdown-menu";
import { ReportContent, ReportContentCode } from "aegis/models";
import { Interactable, InteractableType, Layout, Overflow, Position } from "twitch-core-ui";

import "./styles.scss";

export type Props = {
  defaultContentCode?: ReportContentCode;
  contents: ReportContent[];
  onSelect: (content: ReportContent) => void;
};

export enum TestSelectors {
  Option = "report-content-dropdown--option",
  Toggle = "report-content-dropdown--toggle"
}

@observer
export class ReportContentDropdownComponent extends React.Component<Props> {
  @observable private selectedValue?: ReportContent;
  @observable private show: boolean = false;

  constructor(props: Props) {
    super(props);
    const { defaultContentCode, contents } = this.props;
    if (defaultContentCode) {
      const matched = contents.filter(c => c.code === defaultContentCode);
      if (matched.length > 0) {
        this.selectedValue = matched[0];
      }
    }
  }

  public render() {
    const { contents } = this.props;

    let display: JSX.Element;
    if (this.selectedValue) {
      display = <Layout className="aegis-pill__content">{this.selectedValue.displayName}</Layout>;
    } else {
      display = <Layout className="aegis-pill__content">Click Me</Layout>;
    }

    const list = contents.map(content => (
      <Layout padding={{ top: 0.5 }} key={content.code}>
        <Interactable
          data-track-click={`report-content-dropdown-select-${content.code}`}
          onClick={() => {
            this.onClick(content);
          }}
          data-test-selector={TestSelectors.Option}
        >
          <Layout className="aegis-pill__content">{content.displayName}</Layout>
        </Interactable>
      </Layout>
    ));

    return (
      <ClickOutDetector onClickOut={this.hide}>
        <Layout position={Position.Relative}>
          <Interactable
            data-track-click="report-content-dropdown-show"
            onClick={this.toggleShow}
            type={InteractableType.Alt}
            data-test-selector={TestSelectors.Toggle}
          >
            {display}
          </Interactable>
          <DropdownMenu show={this.show}>
            <Layout className="report-content-dropdown" overflow={Overflow.Scroll}>
              {list}
            </Layout>
          </DropdownMenu>
        </Layout>
      </ClickOutDetector>
    );
  }

  @action
  private hide = () => {
    this.show = false;
  };

  @action
  private toggleShow = () => {
    this.show = !this.show;
  };

  @action
  private onClick = (content: ReportContent) => {
    this.selectedValue = content;
    this.props.onSelect(content);
    this.show = false;
  };
}
