import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { ClickOutDetector } from "aegis/functionality/components/click-out-detector";
import { DropdownMenu } from "aegis/functionality/components/dropdown-menu";
import { ReportReason, ReportReasonCode } from "aegis/models";
import { Interactable, InteractableType, Layout, Overflow, Position } from "twitch-core-ui";

import "./styles.scss";

export type Props = {
  defaultReasonCode?: ReportReasonCode;
  reasons: ReportReason[];
  onSelect: (content: ReportReason) => void;
};

export enum TestSelectors {
  Option = "report-reason-dropdown--option",
  Toggle = "report-reason-dropdown--toggle"
}

@observer
export class ReportReasonDropdownComponent extends React.Component<Props> {
  @observable private selectedValue?: ReportReason;
  @observable private show: boolean = false;

  constructor(props: Props) {
    super(props);
    const { defaultReasonCode, reasons } = this.props;
    if (defaultReasonCode) {
      const matched = reasons.filter(r => r.code === defaultReasonCode);
      if (matched.length > 0) {
        this.selectedValue = matched[0];
      }
    }
  }

  public render() {
    const { reasons } = this.props;

    let display: JSX.Element;
    if (this.selectedValue) {
      display = <Layout className="aegis-pill__reason">{this.selectedValue.displayName}</Layout>;
    } else {
      display = <Layout className="aegis-pill__reason">Click Me</Layout>;
    }

    const list = reasons.map(reason => (
      <Layout padding={{ top: 0.5 }} key={reason.code}>
        <Interactable
          data-track-click={`report-reason-dropdown-select-${reason.code}`}
          onClick={() => {
            this.onClick(reason);
          }}
          data-test-selector={TestSelectors.Option}
        >
          <Layout className="aegis-pill__reason">{reason.displayName}</Layout>
        </Interactable>
      </Layout>
    ));

    return (
      <ClickOutDetector onClickOut={this.hide}>
        <Layout position={Position.Relative}>
          <Interactable
            data-track-click="report-reason-dropdown-show"
            onClick={this.toggleShow}
            type={InteractableType.Alt}
            data-test-selector={TestSelectors.Toggle}
          >
            {display}
          </Interactable>
          <DropdownMenu show={this.show}>
            <Layout className="report-reason-dropdown" overflow={Overflow.Scroll}>
              {list}
            </Layout>
          </DropdownMenu>
        </Layout>
      </ClickOutDetector>
    );
  }

  @action
  private hide = () => {
    this.show = false;
  };

  @action
  private toggleShow = () => {
    this.show = !this.show;
  };

  @action
  private onClick = (content: ReportReason) => {
    this.selectedValue = content;
    this.props.onSelect(content);
    this.show = false;
  };
}
