import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import { ReportQueueActionContext, withReportQueueActionContext } from "aegis/context/report";
import { Report } from "aegis/features/reports";
import { ReportHeartbeatTracker, ReportInViewTracker } from "aegis/features/tracking";
import { User } from "aegis/models";
import { ReportStore, ReportStoreProps } from "aegis/stores";
import { ReportMutationsContextWrapper } from "./components/mutations";
import { ReportHotkeyLayer } from "./components/report-hotkey-layer";
import { ReportActionContextProvider } from "./context";
import { ReportViewType } from "./models";

export interface PublicProps {
  report: Report;
  viewType: ReportViewType;
  reportStore: ReportStore;
  disableExpand?: boolean;
  prevReport?: () => void;
  nextReport?: () => void;
  hideReport?: () => void;
}

type Props = PublicProps & ReportQueueActionContext & ReportStoreProps;

@observer
export class ReportViewComponent extends React.Component<Props> {
  @observable private strikeUser?: User;

  public render() {
    const { report, onAutoResolve, onCancelHold } = this.props;
    return (
      <div data-track-report={report.id}>
        <ReportInViewTracker report={report}>
          <ReportHeartbeatTracker report={report}>
            <AncestorNameContextProvider ancestorName={AncestorName.ReportCard}>
              <ReportActionContextProvider
                {...this.props}
                openStrikeForm={this.openStrikeForm}
                closeStrikeForm={this.closeStrikeForm}
                onAutoResolve={onAutoResolve}
                onCancelHold={onCancelHold ? () => onCancelHold(report) : undefined}
              >
                <ReportMutationsContextWrapper report={report}>
                  <ReportHotkeyLayer {...this.props} strikeUser={this.strikeUser} />
                </ReportMutationsContextWrapper>
              </ReportActionContextProvider>
            </AncestorNameContextProvider>
          </ReportHeartbeatTracker>
        </ReportInViewTracker>
      </div>
    );
  }

  @action
  private openStrikeForm = (user: User) => {
    this.strikeUser = user;
  };

  @action
  private closeStrikeForm = () => {
    this.strikeUser = undefined;
  };
}

function mapContextToProps(context: ReportQueueActionContext): ReportQueueActionContext {
  return context;
}

export const ReportView = withReportQueueActionContext<ReportQueueActionContext, PublicProps>(mapContextToProps)(
  ReportViewComponent
);
