import * as React from "react";

import { inject } from "mobx-react";
import { compose, FetchResult, graphql } from "react-apollo";

import { Report } from "aegis/features/reports";
import { CreateReportComment, EventName } from "aegis/features/tracking";
import { TrackingClientProps } from "aegis/features/tracking/client";
import { ReportMutationContextProvider } from "./context";
import { CreateReportCommentInput, CreateReportCommentResponse, MAKE_COMMENT } from "./create-report-comment";

export interface ComposedMutations {
  createReportComment: (
    input: { variables: CreateReportCommentInput }
  ) => Promise<FetchResult<CreateReportCommentResponse>>;
}

export interface InputProps {
  report: Report;
}

@inject("trackingClient")
export class ReportMutationsContextWrapperComponent extends React.Component<
  ComposedMutations & InputProps & TrackingClientProps
> {
  public render() {
    const { report } = this.props;
    const createComment = (comment: string, ancestorName: string, onComplete?: () => void) => {
      this.props
        .createReportComment({ variables: { input: { reportID: report.id, comment: comment } } })
        .then(({ data }) => {
          if (ancestorName && data && this.props.trackingClient) {
            const event: CreateReportComment = {
              report_id: report.id,
              comment_id: data.createReportComment.reportComment.id,
              ui_context: ancestorName
            };

            this.props.trackingClient.Track(EventName.CreateReportComment, event);
          }
        })
        .then(() => {
          if (onComplete) {
            onComplete();
          }
        });
    };
    return (
      <ReportMutationContextProvider createComment={createComment}>{this.props.children}</ReportMutationContextProvider>
    );
  }
}

export const ReportMutationsContextWrapper: React.ComponentClass<InputProps> = compose(
  graphql(MAKE_COMMENT, {
    name: "createReportComment"
  })
)(ReportMutationsContextWrapperComponent);
