import * as React from "react";

import { shallow } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { ReportBannerComponent, TestSelectors } from "aegis/features/report-view/components/report-banner/component";
import { ReportAudit } from "aegis/features/reports";
import { MockAdmin } from "aegis/models/mocks";
import { mockReport } from "../../reportMocks";

import "aegis/functionality/date-extensions";

Enzyme.configure({ adapter: new Adapter() });

describe("ReportBannerComponent", () => {
  const fakeAudit: ReportAudit = {
    id: "auditID",
    action: "assign_to_admin",
    actionBy: MockAdmin(),
    diff: "",
    createdAt: new Date("2019-01-02T03:04:05Z")
  };

  it("returns null with no audits", () => {
    const wrapper = shallow(<ReportBannerComponent report={mockReport()} />);

    expect(wrapper.children().length).toBe(0);
  });

  it("returns assignment banner with assigned audit", () => {
    const wrapper = shallow(<ReportBannerComponent report={mockReport({ audits: [fakeAudit] })} />);

    expect(wrapper.children().length).toBe(1);
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Assignee })
        .childAt(0)
        .text()
    ).toBe("ldap");
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Assigner })
        .childAt(0)
        .text()
    ).toBe("ldap");
  });

  it("returns previous hold banner with expired hold", () => {
    const fakeHoldAudit: ReportAudit = {
      ...fakeAudit,
      action: "hold",
      hold: {
        id: "id",
        createdBy: MockAdmin(),
        until: new Date("4:20 4/20/1970"),
        createdAt: new Date("3:20 4/20/1970"),
        updatedAt: new Date()
      }
    };
    const wrapper = shallow(<ReportBannerComponent report={mockReport({ audits: [fakeHoldAudit] })} />);

    expect(wrapper.children().length).toBe(1);
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Message })
        .childAt(0)
        .text()
    ).toBe("Previously held by ldap for 60 minutes, on 2019-01-02 03:04:05Z");
  });

  it("returns countdown hold banner when hold isn't expired", () => {
    const now = new Date();
    const later = new Date();
    later.setTime(now.getTime() + 60 * 60 * 1000);
    const fakeHoldAudit: ReportAudit = {
      ...fakeAudit,
      action: "hold",
      hold: { id: "id", createdBy: MockAdmin(), until: later, createdAt: now, updatedAt: now }
    };
    const wrapper = shallow(<ReportBannerComponent report={mockReport({ audits: [fakeHoldAudit] })} />);

    expect(wrapper.children().length).toBe(1);
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Message })
        .childAt(0)
        .childAt(0)
        .text()
    ).toBe("Report held by ldap for 60 minutes, ");
    expect(wrapper.find({ "data-test-selector": TestSelectors.Countdown }).length).toBe(1);
  });
});
