import * as React from "react";

import Countdown from "react-countdown-now";
import { Delta } from "react-countdown-now";

import { Report, ReportHold } from "aegis/features/reports";
import {
  AlignItems,
  Background,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  StyledLayout,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  report: Report;
  disableHold?: () => void;
}

export enum TestSelectors {
  Assigner = "report-banner-assigner",
  Assignee = "report-banner-assignee",
  Queue = "report-banner-queue",
  Message = "report-banner-message",
  Countdown = "report-banner-countdown"
}

export class ReportBannerComponent extends React.Component<Props> {
  public render() {
    const { report } = this.props;
    const { audits } = report;
    if (audits.length === 0) {
      return null;
    }
    const bannerContent = this.getBannerContent();
    return bannerContent ? (
      <StyledLayout
        fullWidth
        padding={{ x: 2, bottom: 1 }}
        color={Color.Alt}
        background={Background.Alt2}
        className="report-card-banner"
        border
      >
        {bannerContent}
      </StyledLayout>
    ) : null;
  }

  private getBannerContent = (): JSX.Element | null => {
    const { report, disableHold } = this.props;
    const { audits } = report;

    // Gets most recent hold or hold_disable
    const holdAction = audits.find(a => {
      return a.action === "hold" || a.action === "hold_disabled";
    });

    // return early if there's an active hold with no earlier hold disabled
    if (holdAction && holdAction.action === "hold" && holdAction.hold && holdAction.actionBy) {
      const holdUntil = new Date(holdAction.hold!.until).getTime();
      if (holdUntil - Date.now() > 0) {
        const minutes = getHoldMinutes(holdAction.hold);
        return (
          <Layout display={Display.Flex} justifyContent={JustifyContent.Between} alignItems={AlignItems.Center}>
            <Layout padding={{ y: 1 }} data-test-selector={TestSelectors.Message}>
              <CoreText lines={1} type={TextType.Span}>
                {`Report held by ${holdAction.actionBy.ldap} for ${minutes} minutes, `}
                <Countdown
                  date={new Date(holdAction.hold.until)}
                  daysInHours
                  renderer={InlineCountdownRenderer}
                  precision={3}
                  onComplete={() => this.forceUpdate()}
                  data-test-selector={TestSelectors.Countdown}
                />
                {" remaining"}
              </CoreText>
            </Layout>
            <Button
              data-track-click="report-banner-disable-hold"
              type={ButtonType.Hollow}
              onClick={e => {
                e.stopPropagation();
                return disableHold && disableHold();
              }}
            >
              <CoreText color={Color.Overlay}>Cancel hold</CoreText>
            </Button>
          </Layout>
        );
      }
    }
    const audit = audits[0];
    if (!audit || !audit.actionBy) {
      return null;
    }

    const timestamp = new Date(audit.createdAt).toDisplayFormat();

    switch (audit.action) {
      case "assign_to_admin": {
        const assignee = (
          <CoreText type={TextType.Span} bold italic data-test-selector={TestSelectors.Assignee}>
            {report.assignedto ? report.assignedto.ldap : "Nobody"}
          </CoreText>
        );

        const assigner = (
          <CoreText type={TextType.Span} bold italic data-test-selector={TestSelectors.Assigner}>
            {`${audit.actionBy.ldap}`}
          </CoreText>
        );

        return (
          <Layout padding={{ y: 1 }}>
            <CoreText type={TextType.Span}>
              Report assigned to {assignee} by {assigner}, on {timestamp}
            </CoreText>
          </Layout>
        );
      }
      case "assign_to_queue": {
        const assigner = (
          <CoreText type={TextType.Span} bold italic data-test-selector={TestSelectors.Assigner}>
            {`${audit.actionBy.ldap}`}
          </CoreText>
        );

        return (
          <Layout padding={{ y: 1 }} data-test-selector={TestSelectors.Message}>
            <CoreText type={TextType.Span}>
              Report moved by {assigner}, on {timestamp}
            </CoreText>
          </Layout>
        );
      }
      case "check_later": {
        const assigner = (
          <CoreText type={TextType.Span} bold italic data-test-selector={TestSelectors.Assigner}>
            {`${audit.actionBy.ldap}`}
          </CoreText>
        );

        return (
          <Layout padding={{ y: 1 }} data-test-selector={TestSelectors.Message}>
            <CoreText type={TextType.Span}>
              Report moved to Investigative Queue by {assigner}, on {timestamp}
            </CoreText>
          </Layout>
        );
      }
      case "hold": {
        // Should only get here if this hold has expired (early returns above for active holds)
        if (!audit.hold) {
          return null;
        }
        const minutes = getHoldMinutes(audit.hold);
        return (
          <Layout padding={{ y: 1 }}>
            <CoreText lines={1} type={TextType.Span} data-test-selector={TestSelectors.Message}>
              {`Previously held by ${audit.actionBy.ldap} for ${minutes} minutes, on ${timestamp}`}
            </CoreText>
          </Layout>
        );
      }
      case "hold_disabled": {
        if (!audit.hold) {
          return null;
        }
        return (
          <Layout padding={{ y: 1 }}>
            <CoreText lines={1} type={TextType.Span} data-test-selector={TestSelectors.Message}>
              {`Hold by ${audit.hold.createdBy.ldap} disabled by ${audit.actionBy.ldap}, on ${timestamp}`}
            </CoreText>
          </Layout>
        );
      }
      default: {
        return null;
      }
    }
  };
}

export function getHoldMinutes(hold: ReportHold) {
  const until = new Date(hold.until);
  const createdAt = new Date(hold.createdAt);

  const diffInMs = until.getTime() - createdAt.getTime();
  const diffInMinutes = Math.round((diffInMs % 86400000) / 60000);

  return diffInMinutes;
}

// Custom countdown renderer to ignore hours
const InlineCountdownRenderer = (delta: Delta) => {
  return (
    <CoreText lines={1} type={TextType.Span}>
      {("0" + delta.minutes).slice(-2)}:{("0" + delta.seconds).slice(-2)}
    </CoreText>
  );
};
