import * as React from "react";

import { Report } from "aegis/features/reports";
import { User } from "aegis/features/user";
import { UserRoleTags } from "aegis/features/user-role-tags";
import { User as modelsUser } from "aegis/models";
import { ReportActionContext, reportActionContextMapper, withReportActionContext } from "../../context";
import { ReportComments } from "../report-comments";
import { ReportEmbed } from "../report-embeds";
import { StreamEmbed } from "./stream-embed";
import { UserMetadata } from "./user-metadata";

import {
  AlignItems,
  Background,
  BalloonDirection,
  BorderRadius,
  Button,
  ButtonType,
  Column,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  Grid,
  InjectLayout,
  Interactable,
  InteractableType,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
  TextTransform,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  report: Report;
  onStrikeFromUser: (User: modelsUser) => void;
}

export class ReportBodyComponent extends React.Component<Props & ReportActionContext> {
  public render() {
    const { report, onStrikeFromUser } = this.props;
    const createdTime = new Date(report.createdAt);
    const reportUserContextButton = report.targetUser && (
      <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
        <Layout display={Display.Inline} padding={{ right: 0.5 }}>
          <CoreLink linkTo={`/altlookup?id=${report.targetUser.id}`} targetBlank>
            <Button data-track-click="report-body-alt-lookup" type={ButtonType.Hollow} tabIndex={-1}>
              <Layout display={Display.Flex}>
                Alt Lookup
                <Layout padding={{ left: 0.5 }} display={Display.Flex}>
                  <SVG asset={SVGAsset.Popout} width={16} height={16} />
                </Layout>
              </Layout>
            </Button>
          </CoreLink>
        </Layout>
        <CoreLink
          linkTo={`/chatlogs/?userids=${report.targetUser.id}&after=${createdTime.getTime() -
            3600000}&before=${createdTime.getTime() + 900000}`}
          targetBlank
        >
          <Button data-track-click="report-body-see-chatlogs" type={ButtonType.Hollow} tabIndex={-1}>
            <Layout display={Display.Flex}>
              See chatlogs
              <Layout padding={{ left: 0.5 }} display={Display.Flex}>
                <SVG asset={SVGAsset.Popout} width={16} height={16} />
              </Layout>
            </Layout>
          </Button>
        </CoreLink>
      </Layout>
    );

    const reportFromUserContext = report.fromUser && (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column} alignItems={AlignItems.Start}>
        <User user={report.fromUser} className="reporter-user-panel" balloonDirection={BalloonDirection.BottomCenter} />
        <UserRoleTags roles={report.fromUser.roles} />
        <Layout fullWidth={false} flexGrow={0}>
          <Interactable
            data-track-click="report-body-strike-from-user"
            onClick={() => {
              onStrikeFromUser(report.fromUser);
            }}
            type={InteractableType.Alpha}
            borderRadius={BorderRadius.Medium}
          >
            <Layout display={Display.Flex} padding={{ top: 0.5, bottom: 0.5 }}>
              <CoreText type={TextType.Span} fontSize={FontSize.Size6}>
                Strike reporter
              </CoreText>
              <SVG asset={SVGAsset.AngleRight} width={15} height={15} />
            </Layout>
          </Interactable>
        </Layout>
      </Layout>
    );

    const streamPreview = report.targetUser && <StreamEmbed roundedCorner id={report.targetUser.id} />;

    return (
      <StyledLayout borderLeft borderRight position={Position.Relative}>
        <StyledLayout
          background={Background.Alt2}
          fullWidth
          padding={{ x: 2, y: 1 }}
          borderTop
          borderBottom
          className="section-header"
          position={Position.Absolute}
        >
          <CoreText bold>INVESTIGATION</CoreText>
        </StyledLayout>
        <Layout fullWidth padding={{ x: 2, top: 4, bottom: 3 }}>
          <Grid justifyContent={JustifyContent.Start}>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <CoreText fontSize={FontSize.Size5}>
                  Assigned To: {report.assignedto ? report.assignedto.ldap : "nobody"}
                </CoreText>
              </Layout>
            </Column>
            <Column cols={{ default: 6, xs: 6, sm: 6, md: 4, lg: 4, xl: 2, xxl: 2 }}>
              <Layout margin={{ y: 1 }}>
                <CoreText type={TextType.Span} fontSize={FontSize.Size5}>
                  Status:{" "}
                </CoreText>
                <CoreText type={TextType.Span} fontSize={FontSize.Size5} transform={TextTransform.Capitalize}>
                  {report.status}
                </CoreText>
              </Layout>
            </Column>
          </Grid>
          <StyledLayout borderBottom fullWidth className="zero-height-bar" margin={{ right: 1, bottom: 2 }} />
          <Layout display={Display.Flex} fullWidth justifyContent={JustifyContent.Between}>
            <Layout display={Display.Flex} padding={{ right: 1, bottom: 2 }} flexGrow={1}>
              <Layout fullWidth flexDirection={FlexDirection.Column}>
                <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                  <Layout flexGrow={1}>
                    <CoreText type={TextType.Span} fontSize={FontSize.Size5} bold lines={1}>
                      Reported User
                    </CoreText>
                  </Layout>
                  <StyledLayout borderBottom fullWidth className="zero-height-bar" margin={{ left: 1 }} />
                </Layout>
                <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} padding={{ right: 1 }}>
                  {report.targetUser && (
                    <User user={report.targetUser} className="reported-user-panel">
                      {reportUserContextButton}
                    </User>
                  )}
                </Layout>
                {report.targetUser && <UserMetadata id={report.targetUser.id} />}
              </Layout>
            </Layout>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
              <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                <CoreText type={TextType.Span} fontSize={FontSize.Size5} bold>
                  Reporter
                </CoreText>
                <StyledLayout borderBottom fullWidth className="zero-height-bar" margin={{ left: 1 }} />
              </Layout>
              {reportFromUserContext}
            </Layout>
          </Layout>
          <Layout fullWidth>{streamPreview}</Layout>
          <Layout
            display={Display.Flex}
            className="reported-content-panel"
            flexDirection={FlexDirection.Column}
            padding={{ y: 1 }}
          >
            <InjectLayout display={Display.Flex} alignItems={AlignItems.Start} fullWidth overflow={Overflow.Visible}>
              <ReportEmbed report={report} />
            </InjectLayout>
          </Layout>
        </Layout>
        <StyledLayout
          background={Background.Alt2}
          fullWidth
          padding={{ x: 2, y: 1 }}
          borderTop
          borderBottom
          className="section-header"
          position={Position.Absolute}
        >
          <CoreText bold>ACTIONS</CoreText>
        </StyledLayout>
        <Layout fullWidth padding={{ top: 3, bottom: 1 }}>
          <ReportComments report={report} />
        </Layout>
      </StyledLayout>
    );
  }
}

export const ReportBody = withReportActionContext<ReportActionContext, Props>(reportActionContextMapper)(
  ReportBodyComponent
);
