import * as React from "react";

import {
  BorderRadius,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  StyledLayout,
  SVGAsset,
  TextType,
  Tooltip
} from "twitch-core-ui";

import { DetailedUser } from "aegis/models";

export type Props = {
  user: DetailedUser;
  onClose?: () => void;
  roundedBottom?: boolean;
};

export enum TestSelectors {
  CloseButton = "stream-embed--close-button",
  Uptime = "stream-embed--uptime"
}

export const StreamEmbedComponent: React.SFC<Props> = ({ onClose, user, roundedBottom }) => {
  const closeButton = (
    <Layout padding={{ left: 0.5 }}>
      <Tooltip label="Hide Stream Preview">
        <Button
          data-track-click="report-body-hide-stream-preview"
          onClick={onClose}
          icon={SVGAsset.Close}
          type={ButtonType.Hollow}
          data-test-selector={TestSelectors.CloseButton}
        />
      </Tooltip>
    </Layout>
  );

  let title: string | null = null;
  let uptime: string | null = null;
  let game: string = "?";
  if (user.channel) {
    game = user.channel.game || "-";
    title = user.channel.title;
  }
  if (user.stream) {
    title = user.stream.title;
    uptime = formatDurationTimestamp(user.stream.duration);
  }

  return (
    <StyledLayout
      border
      borderRadius={
        roundedBottom
          ? {
              bottomRight: BorderRadius.Medium,
              bottomLeft: BorderRadius.Medium
            }
          : undefined
      }
      display={Display.Flex}
    >
      <iframe src={user.streamEmbedURL} width={480} height={270} allowFullScreen />
      <Layout padding={{ x: 1 }} display={Display.Flex} flexDirection={FlexDirection.Column}>
        <InfoField header="Channel" value={user.login} />
        {title && <InfoField header="Title" value={title} />}
        {game && <InfoField header="Game" value={game} />}
        {uptime && <InfoField header="Uptime" value={uptime} data-test-selector={TestSelectors.Uptime} />}
      </Layout>
      {onClose && closeButton}
    </StyledLayout>
  );
};

const InfoField: React.SFC<{ header: string; value: string }> = ({ header, value }) => {
  return (
    <Layout margin={{ y: 0.5 }}>
      <CoreText type={TextType.Strong}>{header}</CoreText>
      <CoreText>{value}</CoreText>
    </Layout>
  );
};

export const formatDurationTimestamp = (seconds: number): string => {
  const h = Math.floor(seconds / 3600);
  const m = Math.floor((seconds % 3600) / 60);
  const s = seconds % 60;
  const str: string[] = [];
  if (h > 0) {
    str.push(`${h}h`);
  }
  if (m > 0) {
    str.push(`${m}m`);
  }
  if (s > 0) {
    str.push(`${s}s`);
  }
  return str.join(" ");
};
