import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { DetailedUser } from "aegis/models";
import { AlignItems, Color, CoreText, Display, FlexDirection, Layout, Placeholder, StyledLayout } from "twitch-core-ui";
import { UserMetadataComponent } from "./user-metadata";

export interface Response {
  user?: DetailedUser;
}

export interface QueryVals {
  id?: string;
}

export const DETAILED_USER_QUERY = gql`
  query userMetadata($id: ID) {
    user(id: $id) {
      ...UserFields
      ...DetailedUserFields
      suspension {
        ...SuspensionFields
      }
      banException {
        ...BanExceptionFields
      }
      isLiveStreaming
    }
  }
  ${Fragments.UserFields}
  ${Fragments.DetailedUserFields}
  ${Fragments.SuspensionFields}
  ${Fragments.BanExceptionFields}
`;

export interface Props {
  id: string;
}

export class DetailedUserQuery extends Query<Response, QueryVals> {}

export const UserMetadata: React.SFC<Props> = ({ id }) => {
  return (
    <DetailedUserQuery displayName="UserMetadata" query={DETAILED_USER_QUERY} variables={{ id }}>
      {({ loading, data, error }) => {
        if (loading) {
          return (
            <Layout display={Display.Flex} alignItems={AlignItems.Start} flexDirection={FlexDirection.Column}>
              <Placeholder width={100} lineCount={1} />
              <div>
                <Placeholder width={100} lineCount={1} />
                <StyledLayout
                  display={Display.Flex}
                  margin={{ top: 0.5 }}
                  padding={{ top: 0.5 }}
                  flexDirection={FlexDirection.Column}
                  alignItems={AlignItems.Start}
                  borderTop
                >
                  <Placeholder width={200} lineCount={2} />
                  <Placeholder width={200} lineCount={1} />
                </StyledLayout>
              </div>
            </Layout>
          );
        }
        if (error || !data || !data.user) {
          console.error("Error loading user metadata: ", error);
          return <CoreText color={Color.Error}>An error occurred: {error ? error.message : "No user loaded"}</CoreText>;
        }
        return <UserMetadataComponent user={data.user} />;
      }}
    </DetailedUserQuery>
  );
};
