import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockDetailedUser } from "aegis/models/mocks";
import { MockEnforcement, MockSuspension, MockWarning } from "aegis/models/mocks/enforcements";
import { SuspensionLabel, TestSelectors, UserMetadataComponent } from "./user-metadata";

import { FETCH_ENFORCEMENTS } from "aegis/features/enforcement/container";
import "aegis/functionality/date-extensions";
import { wrapWithGQLProvider } from "aegis/tests/graphql";

Enzyme.configure({ adapter: new Adapter() });

describe("Report user metadata", () => {
  it("renders with a regular user", () => {
    const wrapper = Enzyme.mount(<UserMetadataComponent user={MockDetailedUser()} />);

    expect(wrapper).toMatchSnapshot();
  });

  it("renders with live streaming", () => {
    const user = MockDetailedUser({ isLiveStreaming: true });
    const wrapper = Enzyme.mount(<UserMetadataComponent user={user} />);

    expect(wrapper).toMatchSnapshot();
  });

  it("renders with suspension", () => {
    const user = MockDetailedUser({
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        permanent: true
      })
    });
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      warning: MockWarning({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z"
      })
    });
    const apolloMock = [
      {
        request: {
          query: FETCH_ENFORCEMENTS,
          variables: {
            user: MockDetailedUser().id
          }
        },
        result: {
          data: {
            enforcements: enforcement
          }
        }
      }
    ];

    const wrapper = Enzyme.mount(
      wrapWithGQLProvider(<UserMetadataComponent user={user} />, apolloMock).wrappedComponent
    );
    const label = wrapper.find({ "data-test-selector": TestSelectors.SuspensionLabel }).last();

    expect(label.text()).toBe("Permanently Banned");
  });

  it("renders with ban exception", () => {
    const user = MockDetailedUser({
      banException: {
        id: "id",
        reason: "reason",
        createdAt: "2018-05-11T18:29:49Z"
      }
    });
    const wrapper = Enzyme.mount(<UserMetadataComponent user={user} />);

    expect(wrapper).toMatchSnapshot();
  });

  it("always renders with suspension history", () => {
    const wrapper = Enzyme.mount(<UserMetadataComponent user={MockDetailedUser()} />);

    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspensionHistory }).hostNodes()).toHaveLength(1);
  });
});

describe("SuspensionLabel", () => {
  it("renders with no suspension", () => {
    const wrapper = Enzyme.mount(<SuspensionLabel user={{ ...MockDetailedUser(), suspension: undefined }} />);
    const label = wrapper.find({ "data-test-selector": TestSelectors.SuspensionHistory }).last();

    expect(label.text()).toBe("No past suspensions");
  });

  it("renders with expired suspension", () => {
    const createdAt = new Date(new Date().getTime() - 10000);
    const wrapper = Enzyme.mount(
      <SuspensionLabel
        user={{
          ...MockDetailedUser(),
          isBanned: false,
          suspension: MockSuspension({
            id: "suspensionID",
            createdAt: createdAt.toString(),
            permanent: false,
            duration: 10
          })
        }}
      />
    );
    const label = wrapper.find({ "data-test-selector": TestSelectors.SuspensionHistory }).last();

    expect(label.text()).toContain("Suspension Expired: ");
  });

  it("renders with current suspension", () => {
    const createdAt = new Date();
    const wrapper = Enzyme.mount(
      <SuspensionLabel
        user={{
          ...MockDetailedUser(),
          isBanned: true,
          suspension: MockSuspension({ id: "suspensionID", duration: 10000, createdAt: createdAt.toString() })
        }}
      />
    );
    const label = wrapper.find({ "data-test-selector": TestSelectors.SuspensionHistory }).last();

    expect(label.text()).toContain("Suspension Expires: ");
  });
});
