import * as React from "react";

import { BanException } from "aegis/features/ban-exception";
import {
  ReportActionContext,
  reportActionContextMapper,
  withReportActionContext
} from "aegis/features/report-view/context";
import { UserRoleTags } from "aegis/features/user-role-tags";
import { OnEscalateList } from "aegis/features/user/components/on-escalate-list/component";
import { bullet, channelLink, DetailedUser } from "aegis/models";
import {
  AlignItems,
  ChannelStatusIndicator,
  ChannelStatusIndicatorStatus,
  Color,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  Layout,
  StyledLayout,
  SVG,
  SVGAsset,
  TextType
} from "twitch-core-ui";
import { LatestEnforcement } from "./latest-enforement/container";

export interface Props {
  user: DetailedUser;
}

export enum TestSelectors {
  SuspensionHistory = "user-metadata--suspension-history",
  SuspensionLabel = "user-metadata--suspension-label"
}

export const SuspensionLabel: React.SFC<{ user: DetailedUser }> = ({ user }) => {
  const { suspension } = user;
  let label = "No past suspensions";
  let icon = SVGAsset.Check;
  if (!!suspension) {
    if (suspension.permanent) {
      if (user.isBanned || user.isDMCABanned) {
        icon = SVGAsset.Ban;
        label = "Permanently Banned";
      } else {
        label = "Was permanently banned";
      }
    } else if (suspension.duration) {
      const createdDate = new Date(suspension.createdAt);
      const expiresDate = new Date(createdDate.getTime() + suspension.duration);
      if (expiresDate > new Date() && user.isBanned) {
        icon = SVGAsset.Ban;
        label = `Suspension Expires: ${new Date(createdDate.getTime() + suspension.duration).toDisplayFormat()}`;
      } else {
        label = `Suspension Expired: ${new Date(createdDate.getTime() + suspension.duration).toDisplayFormat()}`;
      }
    }
  }

  return (
    <CoreText type={TextType.Span}>
      <Layout display={Display.Flex} alignItems={AlignItems.Center}>
        <SVG asset={icon} />
        <Layout padding={{ left: 0.5 }}>
          <CoreText data-test-selector={TestSelectors.SuspensionHistory}>{label}</CoreText>
        </Layout>
      </Layout>
    </CoreText>
  );
};

const UserMetadata: React.SFC<Props & ReportActionContext> = ({ user }) => {
  const language =
    !!user.language && !!user.channel ? (
      <CoreText color={Color.Alt2} fontSize={FontSize.Size6} noWrap>
        <span key="accountLang" className="semi-bold tw-upcase">
          {user.language}
        </span>
        {` account ${bullet} `}
        <span key="channelLang" className="semi-bold tw-upcase">
          {user.channel.language}
        </span>
        {" channel"}
      </CoreText>
    ) : null;

  const suspensionHistory = !!user.suspension ? (
    <>
      <SuspensionLabel user={user} data-test-selector={TestSelectors.SuspensionLabel} />
      <Layout margin={{ top: 0.5, left: 2 }} className="aegis-pill__reason">
        {user.suspension.reason}
      </Layout>
      <LatestEnforcement user={user.id} />
    </>
  ) : (
    <SuspensionLabel user={user} data-test-selector={TestSelectors.SuspensionLabel} />
  );

  const banException = !!user.banException ? (
    <Layout margin={{ top: 1 }}>
      <BanException banException={user.banException} />
    </Layout>
  ) : null;

  const liveStreamLink = !!user.isLiveStreaming ? (
    <CoreLink targetBlank linkTo={channelLink(user)} type={CoreLinkType.Inherit}>
      <Layout display={Display.Flex} alignItems={AlignItems.Center} flexWrap={FlexWrap.NoWrap} margin={{ top: 1 }}>
        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
          <ChannelStatusIndicator status={ChannelStatusIndicatorStatus.Live} />
          <Layout padding={{ right: 0.5 }} />
          Live now
        </Layout>
        <SVG asset={SVGAsset.LogoGlitch} />
      </Layout>
    </CoreLink>
  ) : (
    <Layout display={Display.Flex} alignItems={AlignItems.Center} flexWrap={FlexWrap.NoWrap} margin={{ top: 0.5 }}>
      <ChannelStatusIndicator status={ChannelStatusIndicatorStatus.Offline} />
      <Layout padding={{ right: 0.5 }} />
      <CoreText>Offline</CoreText>
    </Layout>
  );

  const extraInfo =
    !!liveStreamLink || !!banException ? (
      <StyledLayout>
        {liveStreamLink}
        {banException}
      </StyledLayout>
    ) : null;

  return (
    <Layout display={Display.Flex} alignItems={AlignItems.Start} flexDirection={FlexDirection.Column}>
      {language}
      <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
        <OnEscalateList IsEscalatePartner={user.isEscalatePartner} />
        <UserRoleTags roles={user.roles} />
      </Layout>
      <div>
        {extraInfo}
        <StyledLayout
          display={Display.Flex}
          margin={{ top: 0.5 }}
          padding={{ top: 0.5 }}
          flexDirection={FlexDirection.Column}
          alignItems={AlignItems.Start}
          borderTop
        >
          {suspensionHistory}
        </StyledLayout>
      </div>
    </Layout>
  );
};

export const UserMetadataComponent = withReportActionContext<ReportActionContext, Props>(reportActionContextMapper)(
  UserMetadata
);
