import * as React from "react";

import { Report } from "aegis/features/reports";
import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { AuthStoreProps } from "aegis/stores";
import { inject } from "mobx-react";
import { ASSIGN_REPORT_TO_ADMIN, AssignReportToAdminMutation } from "../../mutations/assign-report-to-admin";
import { InvestigateButtonComponent } from "./component";

export enum ButtonType {
  INVESTIGATE,
  CANCEL
}

export interface InputProps {
  buttonType: ButtonType;
  report: Report;
  onInvestigate?: (report: Report) => void;
  onCancel?: (report: Report) => void;
}

type Props = InputProps & AuthStoreProps;

@inject("authStore")
export class InvestigateButton extends React.Component<Props> {
  public render() {
    const { report, buttonType, authStore, onInvestigate, onCancel } = this.props;

    return (
      <AssignReportToAdminMutation
        mutation={ASSIGN_REPORT_TO_ADMIN}
        onCompleted={data => {
          if (data && data.assignReportToAdmin) {
            if (onInvestigate && data.assignReportToAdmin.report.assignedto) {
              onInvestigate(data.assignReportToAdmin.report);
            } else if (onCancel) {
              onCancel(data.assignReportToAdmin.report);
            }
          }
        }}
        onError={error => console.error("Failed to assign investigate: ", error.message)}
      >
        {(investigateQuery, { loading, error }) => {
          const onSubmit = () => {
            if (authStore!.loggedInUser !== null) {
              investigateQuery({
                variables: {
                  input: {
                    id: report.id,
                    adminLdap: buttonType === ButtonType.INVESTIGATE ? authStore!.loggedInUser!.ldap : undefined
                  }
                }
              });
            }
          };

          // Confirm the action if report is already assigned
          if (report.assignedto && buttonType === ButtonType.INVESTIGATE) {
            return (
              <WithConfirmationModal
                title="Confirm Takeover"
                body={`${
                  report.assignedto.ldap
                } is currently investigating this report. Are you sure you want to take over?`}
                confirmLabelOverride="Investigate"
                onConfirm={onSubmit}
              >
                {confirm => {
                  return (
                    <InvestigateButtonComponent
                      onClick={confirm}
                      error={error ? error.message : undefined}
                      type={buttonType}
                      loading={loading}
                    />
                  );
                }}
              </WithConfirmationModal>
            );
          }

          return (
            <InvestigateButtonComponent
              loading={loading}
              error={error ? error.message : undefined}
              onClick={onSubmit}
              type={buttonType}
            />
          );
        }}
      </AssignReportToAdminMutation>
    );
  }
}
