import * as React from "react";

import { inject } from "mobx-react";

import { Report } from "aegis/features/reports";
import { User } from "aegis/models";
import { AuthStoreProps } from "aegis/stores";
import { Display, FlexWrap, JustifyContent, Layout } from "twitch-core-ui";
import { ButtonType, InvestigateButton } from "../investigate-button";
import { ReassignReportButton } from "../reassign-button";
import { ResolveReportButton } from "../resolve-button";
import { StrikeButton } from "../strike-button";

import { ReportActionContext, withReportActionContext } from "aegis/features/report-view/context";

export interface InputProps {
  report: Report;
  expanded?: boolean;
  onResolve?: (report: Report) => void;
  onReassign?: (report: Report) => void;
  onInvestigate?: (report: Report) => void;
  onCancelInvestigate?: (report: Report) => void;
}

interface ContextProps {
  openStrikeForm: (User?: User) => void;
}

type Props = InputProps & AuthStoreProps & ContextProps;

@inject("authStore")
export class InvestigativeQueueButtonsComponent extends React.Component<Props> {
  public render() {
    const { report, expanded, openStrikeForm, onResolve, onReassign, onCancelInvestigate, onInvestigate } = this.props;
    const resolved = report.status === "resolved";
    const strikeButton = <StrikeButton report={report} onStrike={openStrikeForm} />;
    const resolveButton = <ResolveReportButton report={report} resolved={resolved} onResolve={onResolve} />;
    const reassignButton = <ReassignReportButton report={report} onReassign={onReassign} />;

    if (expanded) {
      return (
        <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} justifyContent={JustifyContent.Between}>
          <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
            {this.isActionEnabled && <Layout padding={{ right: 0.5 }}>{strikeButton}</Layout>}
            {this.isActionEnabled && <Layout padding={{ x: 0.5 }}>{resolveButton}</Layout>}
            <Layout padding={{ left: 0.5 }}>{reassignButton}</Layout>
          </Layout>
          <Layout>
            <InvestigateButton
              report={report}
              buttonType={this.isActionEnabled ? ButtonType.CANCEL : ButtonType.INVESTIGATE}
              onCancel={onCancelInvestigate}
              onInvestigate={onInvestigate}
            />
          </Layout>
        </Layout>
      );
    } else {
      return (
        <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap}>
          <InvestigateButton
            report={report}
            buttonType={this.isActionEnabled ? ButtonType.CANCEL : ButtonType.INVESTIGATE}
            onCancel={onCancelInvestigate}
            onInvestigate={onInvestigate}
          />
        </Layout>
      );
    }
  }

  // Actions are enabeld for investigative queue buttons if report is assigned
  // to the current user
  private get isActionEnabled(): boolean {
    const { report, authStore } = this.props;

    return !!(
      report.assignedto &&
      authStore!.hasLoggedInUser &&
      report.assignedto.ldap === authStore!.loggedInUser!.ldap
    );
  }
}

function mapContextToProps(context: ReportActionContext): ContextProps {
  return { openStrikeForm: context.openStrikeForm };
}

export const InvestigativeQueueButtons = withReportActionContext<ContextProps, InputProps>(mapContextToProps)(
  InvestigativeQueueButtonsComponent
);
