import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { Report, ReportQueue } from "aegis/features/reports";
import * as Fragments from "aegis/fragments";
import { apolloErrorHandler } from "aegis/functionality/error-notifications";
import { ASSIGN_REPORT_TO_ADMIN, AssignReportToAdminMutation } from "../../mutations/assign-report-to-admin";
import { ReassignReportButtonComponent } from "./component";

export const ASSIGN_REPORT_TO_QUEUE_MUTATION = gql`
  mutation AssignToQueue($id: ID!, $queue: ReportQueue!, $language: String!) {
    assignReportToQueue(input: { id: $id, queue: $queue, language: $language }) {
      report {
        ...ReportFields
      }
    }
  }

  ${Fragments.ReportFields}
`;

export type AssignReportInput = {
  id: string;
  language: string;
  queue: ReportQueue;
};

export type AssignReportResponse = {
  assignReportToQueue: {
    report: Report;
  };
};

export class AssignToQueueMutation extends Mutation<AssignReportResponse, AssignReportInput> {}

export interface ReassignButtonProps {
  report: Report;
  onReassign?: (report: Report) => void;
}

export const ReassignReportButton: React.SFC<ReassignButtonProps> = ({ report, onReassign }) => {
  return (
    <AssignToQueueMutation
      mutation={ASSIGN_REPORT_TO_QUEUE_MUTATION}
      onCompleted={assignToQueueData => {
        if (assignToQueueData && assignToQueueData.assignReportToQueue && onReassign) {
          onReassign(assignToQueueData.assignReportToQueue.report);
        }
      }}
      onError={apolloErrorHandler("Failed to reassign")}
    >
      {(assignReportToQueue, assignToQueueResponse) => {
        const assignWithLanguage = (language: string) => {
          const input: AssignReportInput = {
            id: report.id,
            language: language,
            queue: ReportQueue.Investigative
          };
          assignReportToQueue({ variables: input });
        };

        return (
          <AssignReportToAdminMutation
            mutation={ASSIGN_REPORT_TO_ADMIN}
            onCompleted={assignToAdminData => {
              if (assignToAdminData && assignToAdminData.assignReportToAdmin && onReassign) {
                onReassign(assignToAdminData.assignReportToAdmin.report);
              }
            }}
            onError={apolloErrorHandler("Failed to reassign")}
          >
            {(reassignReportQuery, assignToAdminResponse) => {
              const onSubmit = (adminLdap: string) => {
                reassignReportQuery({ variables: { input: { id: report.id, adminLdap: adminLdap } } });
              };

              const anyResponse = assignToAdminResponse.data != null || assignToQueueResponse.data != null;
              const loading = assignToQueueResponse.loading || assignToAdminResponse.loading;

              return (
                <ReassignReportButtonComponent
                  loading={loading}
                  submitReassign={onSubmit}
                  assignToQueueWithLanguage={assignWithLanguage}
                  success={anyResponse}
                  report={report}
                />
              );
            }}
          </AssignReportToAdminMutation>
        );
      }}
    </AssignToQueueMutation>
  );
};
