import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";
import {
  AlignContent,
  AlignItems,
  Background,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexWrap,
  JustifyContent,
  Layout,
  Position,
  StyledLayout,
  TextTransform,
  TextType
} from "twitch-core-ui";

import { LocalesPicker } from "aegis/features/language-lookup-input";
import { ReportCommentInput } from "aegis/features/report-view/components/report-comments/comment-input";
import { Report } from "aegis/features/reports";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import "./styles.scss";

export enum TestSelectors {
  AssignButton = "check-later-modal-button",
  QuickSelectLanguageButton = "check-later-quick-select-language-button"
}

export const QuickSelectButton: React.SFC<{ language: string; onLanguageSelect: (language: string) => void }> = ({
  language,
  onLanguageSelect
}) => (
  <Layout padding={{ right: 1 }}>
    <Button
      data-track-click={`report-assign-to-investigative-quick-select-${language}`}
      type={ButtonType.Hollow}
      onClick={() => {
        onLanguageSelect(language);
      }}
      data-test-selector={TestSelectors.QuickSelectLanguageButton}
    >
      <CoreText transform={TextTransform.Uppercase}>{language}</CoreText>
    </Button>
  </Layout>
);

interface Props {
  closeModal?: () => void;
  report: Report;
  assignToQueueWithLanguage: (language: string) => void;
}

@observer
export class CheckLaterModalComponent extends React.Component<Props> {
  @observable language: string | null = "en";

  public constructor(props: Props) {
    super(props);
    if (props.report.language) {
      this.onLanguageSelect(props.report.language);
    }
  }

  public render() {
    return (
      <AncestorNameContextProvider ancestorName={AncestorName.InvestigativeQueueModal}>
        <StyledLayout className="check-later-modal" alignContent={AlignContent.Center} alignItems={AlignItems.Center}>
          <StyledLayout background={Background.Base} fullHeight={true} alignContent={AlignContent.Center} padding={1}>
            <Layout display={Display.Flex} justifyContent={JustifyContent.Between}>
              <CoreText type={TextType.H3}>Assign to the Investigative Queue</CoreText>
            </Layout>
            <Layout padding={{ top: 2 }}>
              <CoreText>Add a comment</CoreText>
              <ReportCommentInput />
            </Layout>
            <Layout padding={{ top: 1 }} position={Position.Relative}>
              <Layout
                display={Display.Flex}
                flexWrap={FlexWrap.NoWrap}
                alignItems={AlignItems.Center}
                justifyContent={JustifyContent.Start}
              >
                <CoreText type={TextType.Span}>Language selected:</CoreText>
                <Layout padding={{ right: 0.5 }} />
                <CoreText bold type={TextType.H4} transform={TextTransform.Uppercase}>
                  <Layout display={Display.InlineBlock}>{this.language}</Layout>
                </CoreText>
              </Layout>
              <Layout
                display={Display.Flex}
                flexWrap={FlexWrap.NoWrap}
                alignItems={AlignItems.Center}
                justifyContent={JustifyContent.Start}
              >
                <CoreText type={TextType.Span}>Quick select:</CoreText>
                <Layout padding={{ right: 0.5 }} />
                <QuickSelectButton onLanguageSelect={this.onLanguageSelect} language={"en"} />
                <QuickSelectButton onLanguageSelect={this.onLanguageSelect} language={"ru"} />
                <QuickSelectButton onLanguageSelect={this.onLanguageSelect} language={"de"} />
                <QuickSelectButton onLanguageSelect={this.onLanguageSelect} language={"tr"} />
                <QuickSelectButton onLanguageSelect={this.onLanguageSelect} language={"ko"} />
              </Layout>
              <LocalesPicker onSelect={this.onLanguageSelect} />
            </Layout>
            <Layout padding={{ top: 1 }} fullWidth display={Display.Flex} justifyContent={JustifyContent.Center}>
              <Button
                data-track-click="report-to-investigative-submit"
                onClick={() => {
                  // TODO: Determine if 'null' case makes sense here
                  this.props.assignToQueueWithLanguage(this.language || "en");
                }}
                data-test-selector={TestSelectors.AssignButton}
              >
                <Layout className="assign-button">Assign</Layout>
              </Button>
            </Layout>
          </StyledLayout>
        </StyledLayout>
      </AncestorNameContextProvider>
    );
  }

  @action
  private onLanguageSelect = (language: string | null) => {
    this.language = language;
  };
}
