import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { ReportQueueActionContext } from "aegis/context/report";
import {
  ReportActionContext,
  reportActionContextMapper,
  withReportActionContext
} from "aegis/features/report-view/context";
import { AutoResolve } from "aegis/models";
import { AutoResolveButtonComponent } from "./component";

export const CREATE_AUTO_RESOLVE = gql`
  mutation CREATE_AUTO_RESOLVE($input: CreateAutoResolveRuleInput!) {
    createAutoResolveRule(input: $input) {
      autoResolve {
        id
      }
    }
  }
`;

type CreateAutoResolveReportInput = {
  input: {
    fromUser?: CreateAutoResolveUser;
    targetUser?: CreateAutoResolveUser;
    content?: string;
    reason?: string;
    seconds: number;
  };
};

type CreateAutoResolveUser = {
  id: string;
  type: string;
};

type CreateAutoResolveResponse = {
  createAutoResolveRule: {
    autoResolve: AutoResolve;
  };
};

export interface Props {
  fromUser?: CreateAutoResolveUser;
  targetUser?: CreateAutoResolveUser;
  content?: string;
  reason?: string;
  seconds: number;
  onSuccess?: (autoResolve: AutoResolve) => void;
}

export class CreateAutoResolveMutation extends Mutation<CreateAutoResolveResponse, CreateAutoResolveReportInput> {}

export const AutoResolveButtonContainer: React.SFC<Props & ReportQueueActionContext> = ({
  fromUser,
  targetUser,
  content,
  reason,
  seconds,
  onSuccess,
  onAutoResolve
}) => {
  const onCompleted = (data: CreateAutoResolveResponse) => {
    if (onSuccess) {
      onSuccess(data.createAutoResolveRule.autoResolve);
    }
    if (onAutoResolve) {
      onAutoResolve();
    }
  };

  return (
    <CreateAutoResolveMutation
      mutation={CREATE_AUTO_RESOLVE}
      onError={error => console.error("Failed to create auto resolve: ", error.message)}
      onCompleted={onCompleted}
    >
      {(createAutoResolve, { loading, error }) => {
        const onClick = () => {
          createAutoResolve({
            variables: {
              input: {
                fromUser: fromUser,
                targetUser: targetUser,
                reason: reason,
                content: content,
                seconds: seconds
              }
            }
          });
        };

        let disabledReason: string | undefined;

        if (!content && !reason) {
          disabledReason = "One of content or reason must be specified";
        } else if (!fromUser && !targetUser) {
          disabledReason = "One of target or from user must be specified";
        }

        return (
          <AutoResolveButtonComponent
            disabled={loading || !!disabledReason}
            tooltip={!!disabledReason ? disabledReason : undefined}
            error={error ? error.message : undefined}
            onClick={onClick}
          />
        );
      }}
    </CreateAutoResolveMutation>
  );
};

export const AutoResolveButton = withReportActionContext<ReportActionContext, Props>(reportActionContextMapper)(
  AutoResolveButtonContainer
);
