import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { FETCH_REPORT_CONTENTS } from "aegis/features/report-content-dropdown";
import { FETCH_REPORT_REASONS } from "aegis/features/report-reason-dropdown";
import { mockReport } from "aegis/features/report-view/reportMocks";
import { MockReportContent, MockReportReason } from "aegis/models/mocks/report";
import { mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { TestProvider } from "aegis/tests/utils/test-provider";
import { AutoResolveButton } from "./auto-resolve-button";
import { Props as AutoResolveComponentProps } from "./auto-resolve-button";
import { AutoResolveCard, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const mockReportContentsSuccess = mockSuccessResponse(
  FETCH_REPORT_CONTENTS,
  {},
  {
    data: {
      reportContents: [MockReportContent()]
    }
  }
);

const mockReportReasonsSuccess = mockSuccessResponse(
  FETCH_REPORT_REASONS,
  {},
  {
    data: {
      reportReasons: [MockReportReason()]
    }
  }
);

const setup = (overrides?: Partial<Props>) => {
  const combinedProps = {
    report: mockReport(),
    show: true,
    ...overrides
  };
  const wrapper = Enzyme.mount(
    wrapWithGQLProvider(
      <TestProvider>
        <AutoResolveCard {...combinedProps} />
      </TestProvider>,
      [mockReportContentsSuccess, mockReportReasonsSuccess]
    ).wrappedComponent
  );
  return {
    props: combinedProps,
    wrapper
  };
};

const findComponentWrapper = (
  wrapper: Enzyme.ReactWrapper<{}, {}>
): Enzyme.ReactWrapper<AutoResolveComponentProps, {}> => {
  const childWrapper = wrapper.find(AutoResolveButton);
  expect(childWrapper).toBeTruthy();
  return childWrapper;
};

describe("Auto Resolve Card", () => {
  it("renders when show", () => {
    const wrapper = Enzyme.shallow(
      <TestProvider>
        <AutoResolveCard show report={mockReport()} />
      </TestProvider>
    );

    expect(wrapper).toBeTruthy();
  });

  it("renders when not show", () => {
    const wrapper = Enzyme.shallow(
      <TestProvider>
        <AutoResolveCard show report={mockReport()} />
      </TestProvider>
    );

    expect(wrapper).toBeTruthy();
  });

  it("updates seconds options correctly", () => {
    const wrapper = setup().wrapper;

    wrapper
      .find({ "data-test-selector": TestSelectors.SecondsOption })
      .first()
      .find({ "data-a-target": "tw-segmented-button-option" })
      .simulate("change");
    expect(findComponentWrapper(wrapper).props().seconds).toBe(5);

    wrapper
      .find({ "data-test-selector": TestSelectors.SecondsOption })
      .last()
      .find({ "data-a-target": "tw-segmented-button-option" })
      .simulate("change");
    expect(findComponentWrapper(wrapper).props().seconds).toBe(30 * 60);
  });

  it("updates target user correctly", () => {
    const report = mockReport();
    const wrapper = setup({ report: report }).wrapper;

    expect(findComponentWrapper(wrapper).props().targetUser).toEqual({
      id: report.targetUser!.id,
      type: "IS"
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.TargetUserCheckbox })
      .find({ "data-a-target": "tw-checkbox" })
      .simulate("change");
    expect(findComponentWrapper(wrapper).props().targetUser).toBe(undefined);
  });

  it("updates from user correctly", () => {
    const report = mockReport();
    const wrapper = setup({ report: report }).wrapper;

    expect(findComponentWrapper(wrapper).props().fromUser).toBe(undefined);

    wrapper
      .find({ "data-test-selector": TestSelectors.FromUserCheckbox })
      .find({ "data-a-target": "tw-checkbox" })
      .simulate("change");

    expect(findComponentWrapper(wrapper).props().fromUser).toEqual({
      id: report.fromUser!.id,
      type: "IS"
    });
  });

  it("updates content correctly", () => {
    const report = mockReport();
    const wrapper = setup({ report: report }).wrapper;

    expect(findComponentWrapper(wrapper).props().content).toBe(report.content.code);

    wrapper
      .find({ "data-test-selector": TestSelectors.ContentCheckbox })
      .find({ "data-a-target": "tw-checkbox" })
      .simulate("change");

    expect(findComponentWrapper(wrapper).props().content).toBe(undefined);
  });

  it("updates reason correctly", () => {
    const report = mockReport();
    const wrapper = setup({ report: report }).wrapper;

    expect(findComponentWrapper(wrapper).props().reason).toBe(undefined);

    wrapper
      .find({ "data-test-selector": TestSelectors.ReasonCheckbox })
      .find({ "data-a-target": "tw-checkbox" })
      .simulate("change");

    expect(findComponentWrapper(wrapper).props().reason).toBe(report.reason.code);
  });
});
