import * as React from "react";

import { action, observable } from "mobx";
import { inject, observer } from "mobx-react";

import { ReportContentDropdown } from "aegis/features/report-content-dropdown";
import { ReportReasonDropdown } from "aegis/features/report-reason-dropdown";
import { Report } from "aegis/features/reports";
import { DropdownMenu } from "aegis/functionality/components/dropdown-menu";
import { AutoResolve, ReportContent, ReportReason, User } from "aegis/models";
import { AutoResolveButton } from "./auto-resolve-button";

import {
  AlignContent,
  AlignItems,
  Balloon,
  BalloonDirection,
  BalloonSize,
  Button,
  ButtonType,
  Card,
  CardBody,
  CheckBox,
  Column,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  Grid,
  Interactable,
  InteractableType,
  JustifyContent,
  Layout,
  Position,
  SegmentedButton,
  SegmentedButtonOption,
  StyledLayout,
  SVG,
  SVGAsset,
  TextAlign,
  TextType,
  VerticalAlign
} from "twitch-core-ui";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import { CreateReportAutoresolve, EventName } from "aegis/features/tracking";
import { TrackingClientProps } from "aegis/features/tracking/client";
import "./styles.scss";

export interface Props {
  show: boolean;
  report: Report;
  onClose?: () => void;
  onStrike?: (User?: User) => void;
}

export enum TestSelectors {
  SecondsOption = "auto-resolve-card--seconds-option",
  FromUserCheckbox = "auto-resolve-card--from-user-checkbox",
  TargetUserCheckbox = "auto-resolve-card--target-user-checkbox",
  ContentCheckbox = "auto-resolve-card--content-checkbox",
  ReasonCheckbox = "auto-resolve-card--reason-checkbox"
}

enum FromUserType {
  FROM = "from",
  NOT_FROM = "not from"
}

@inject("trackingClient")
@observer
export class AutoResolveCard extends React.Component<Props & TrackingClientProps> {
  @observable private showFromUserTypeList: boolean = false;
  @observable private fromUserType: FromUserType = FromUserType.FROM;
  @observable private content?: string;
  @observable private reason?: string;
  @observable private seconds: number = 5;
  @observable private fromUserSelected: boolean = false;
  @observable private targetUserSelected: boolean = true;
  @observable private contentSelected: boolean = true;
  @observable private reasonSelected: boolean = false;

  @action
  componentDidMount() {
    this.content = this.props.report.content.code;
    this.reason = this.props.report.reason.code;
  }

  public render() {
    const { report, show, onClose } = this.props;
    const autoResolveTimeGroup = `auto-resolve-time-${report.id}`;

    const timeOptions: { label: string; value: number; ref: AutoResolveCard }[] = [
      { label: "5 s", value: 5, ref: this },
      { label: "1 min", value: 60, ref: this },
      { label: "5 min", value: 5 * 60, ref: this },
      { label: "10 min", value: 10 * 60, ref: this },
      { label: "15 min", value: 15 * 60, ref: this },
      { label: "30 min", value: 30 * 60, ref: this }
    ];

    const timeOptionsElement = (
      <div className="auto-resolve-card__time-options">
        <SegmentedButton>
          <Grid>
            {timeOptions.map(function(item: { label: string; value: number; ref: AutoResolveCard }, i: number) {
              return (
                <Column cols={4} key={`auto-resolve-time-option-${i}`}>
                  <Layout className="auto-resolve-card-option" padding={{ bottom: 1 }}>
                    <SegmentedButtonOption
                      name={autoResolveTimeGroup}
                      label={item.label}
                      checked={item.ref.seconds === item.value}
                      onChange={() => item.ref.updateSeconds(item.value)}
                      data-test-selector={TestSelectors.SecondsOption}
                    />
                  </Layout>
                </Column>
              );
            })}
          </Grid>
        </SegmentedButton>
      </div>
    );

    const targetUserOptions = report.targetUser && (
      <Layout padding={{ top: 0.5 }}>
        <Layout
          display={Display.InlineFlex}
          justifyContent={JustifyContent.Around}
          alignItems={AlignItems.Center}
          textAlign={TextAlign.Center}
          verticalAlign={VerticalAlign.Middle}
          flexWrap={FlexWrap.NoWrap}
        >
          <CheckBox
            label=""
            checked={this.targetUserSelected}
            onChange={this.toggleTargetUserSelected}
            data-test-selector={TestSelectors.TargetUserCheckbox}
          />

          <Layout display={Display.Inline} padding={{ right: 0.5 }}>
            <CoreText>towards</CoreText>
          </Layout>

          <CoreText bold>{report.targetUser.login}</CoreText>
        </Layout>
      </Layout>
    );

    const fromUserOptions = report.fromUser && (
      <Layout>
        <Layout
          display={Display.InlineFlex}
          justifyContent={JustifyContent.Around}
          alignItems={AlignItems.Center}
          textAlign={TextAlign.Center}
          verticalAlign={VerticalAlign.Middle}
          flexWrap={FlexWrap.NoWrap}
        >
          <CheckBox
            label=""
            checked={this.fromUserSelected}
            onChange={this.toggleFromUserSelected}
            data-test-selector={TestSelectors.FromUserCheckbox}
          />

          <Layout position={Position.Relative} padding={{ right: 0.5 }}>
            <Interactable
              data-track-click="auto-resolve-card-show-from-user"
              onClick={this.toggleShowFromUserList}
              type={InteractableType.Alt}
            >
              <Layout
                display={Display.InlineFlex}
                justifyContent={JustifyContent.Around}
                alignItems={AlignItems.Center}
              >
                {this.fromUserType}
                <SVG asset={SVGAsset.GlyphArrDown} />
              </Layout>
            </Interactable>
            <DropdownMenu show={this.showFromUserTypeList}>
              <Interactable
                data-track-click="auto-resolve-card-from-user-from"
                onClick={() => {
                  this.updateFromUserType(FromUserType.FROM);
                }}
              >
                <Layout
                  className={"auto-resolve-card__user-option"}
                  display={Display.InlineFlex}
                  justifyContent={JustifyContent.Start}
                  padding={{ x: 0.5 }}
                >
                  {FromUserType.FROM}
                </Layout>
              </Interactable>
              <Interactable
                data-track-click={`auto-resolve-card-from-user-not-from`}
                onClick={() => {
                  this.updateFromUserType(FromUserType.NOT_FROM);
                }}
              >
                <Layout
                  className={"auto-resolve-card__user-option"}
                  display={Display.InlineFlex}
                  justifyContent={JustifyContent.Start}
                  padding={{ x: 0.5 }}
                >
                  {FromUserType.NOT_FROM}
                </Layout>
              </Interactable>
            </DropdownMenu>
          </Layout>

          <CoreText bold>{report.fromUser.login}</CoreText>
        </Layout>
      </Layout>
    );

    return (
      <Balloon
        show={show}
        noTail
        size={BalloonSize.Medium}
        offsetY={"0px"}
        direction={BalloonDirection.BottomLeft}
        elevation={1}
      >
        <AncestorNameContextProvider ancestorName={AncestorName.ResolveDropdown}>
          <Card border>
            <CardBody>
              <Layout
                className={"auto-resolve-card"}
                padding={{ x: 0.5, y: 1 }}
                display={Display.Flex}
                flexDirection={FlexDirection.Column}
                fullWidth
              >
                <StyledLayout borderBottom fullWidth padding={{ top: 0.5, bottom: 1 }}>
                  <Layout padding={{ left: 0.5, bottom: 0.5 }}>
                    <CoreText>Automatically resolve reports for:</CoreText>
                  </Layout>
                  {timeOptionsElement}
                  {targetUserOptions}
                  {fromUserOptions}
                </StyledLayout>

                <StyledLayout padding={{ y: 1 }}>
                  <Layout
                    display={Display.InlineFlex}
                    justifyContent={JustifyContent.Around}
                    alignItems={AlignItems.Center}
                    textAlign={TextAlign.Center}
                    verticalAlign={VerticalAlign.Middle}
                    flexWrap={FlexWrap.NoWrap}
                  >
                    <CheckBox
                      label=""
                      checked={this.contentSelected}
                      onChange={this.toggleContentSelected}
                      data-test-selector={TestSelectors.ContentCheckbox}
                    />

                    <CoreText type={TextType.Span}>Restrict to content</CoreText>

                    <Layout padding={{ left: 0.5 }}>
                      <ReportContentDropdown onSelect={this.updateContent} defaultContentCode={report.content.code} />
                    </Layout>
                  </Layout>

                  <Layout
                    display={Display.InlineFlex}
                    justifyContent={JustifyContent.Around}
                    alignItems={AlignItems.Center}
                    textAlign={TextAlign.Center}
                    verticalAlign={VerticalAlign.Middle}
                    flexWrap={FlexWrap.NoWrap}
                    padding={{ y: 0.5 }}
                  >
                    <CheckBox
                      label=""
                      checked={this.reasonSelected}
                      onChange={this.toggleReasonSelected}
                      data-test-selector={TestSelectors.ReasonCheckbox}
                    />

                    <CoreText type={TextType.Span}>Restrict to reason</CoreText>

                    <Layout padding={{ left: 0.5 }}>
                      <ReportReasonDropdown onSelect={this.updateReason} defaultReasonCode={report.reason.code} />
                    </Layout>
                  </Layout>
                </StyledLayout>

                <StyledLayout
                  display={Display.Flex}
                  justifyContent={JustifyContent.Around}
                  alignContent={AlignContent.Center}
                >
                  <Button data-track-click="auto-resolve-card-cancel" type={ButtonType.Hollow} onClick={onClose}>
                    Cancel
                  </Button>
                  <AutoResolveButton
                    fromUser={
                      this.fromUserSelected && report.fromUser
                        ? {
                            id: report.fromUser.id,
                            type: this.fromUserType === FromUserType.FROM ? "IS" : "IS_NOT"
                          }
                        : undefined
                    }
                    targetUser={
                      this.targetUserSelected && report.targetUser
                        ? {
                            id: report.targetUser.id,
                            type: "IS"
                          }
                        : undefined
                    }
                    reason={this.reasonSelected ? this.reason : undefined}
                    content={this.contentSelected ? this.content : undefined}
                    seconds={this.seconds}
                    onSuccess={this.onAutoResolve}
                  />
                </StyledLayout>
              </Layout>
            </CardBody>
          </Card>
        </AncestorNameContextProvider>
      </Balloon>
    );
  }

  @action
  private updateSeconds = (seconds: number) => {
    this.seconds = seconds;
  };

  @action
  private updateContent = (content: ReportContent) => {
    this.content = content.code;
    this.contentSelected = true;
  };

  @action
  private updateReason = (reason: ReportReason) => {
    this.reason = reason.code;
    this.reasonSelected = true;
  };

  @action
  private toggleFromUserSelected = () => {
    this.fromUserSelected = !this.fromUserSelected;
  };

  @action
  private toggleTargetUserSelected = () => {
    this.targetUserSelected = !this.targetUserSelected;
  };

  @action
  private toggleContentSelected = () => {
    this.contentSelected = !this.contentSelected;
  };

  @action
  private toggleReasonSelected = () => {
    this.reasonSelected = !this.reasonSelected;
  };

  @action
  private toggleShowFromUserList = () => {
    this.showFromUserTypeList = !this.showFromUserTypeList;
  };

  @action
  private updateFromUserType = (newValue: FromUserType) => {
    this.fromUserType = newValue;
    this.showFromUserTypeList = false;
  };

  @action
  private onAutoResolve = (autoresolve: AutoResolve) => {
    const { onClose, report, trackingClient } = this.props;

    const event: CreateReportAutoresolve = {
      report_id: report.id,
      autoresolve_rule_id: autoresolve.id
    };

    trackingClient!.Track(EventName.CreateAutoresolveRule, event);

    if (onClose) {
      onClose();
    }
  };
}
