import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { Report } from "aegis/features/reports";
import { ClickOutDetector } from "aegis/functionality/components/click-out-detector";
import { CoreButtonType, Layout, Position, SplitButton, Tooltip } from "twitch-core-ui";
import { AutoResolveCard } from "./auto-resolve-card/component";

export interface ResolveButtonComponentProps {
  report: Report;
  loading: boolean;
  success?: boolean;
  error?: string;
  onClick: () => void;
}

@observer
export class ResolveReportButtonComponent extends React.Component<ResolveButtonComponentProps> {
  @observable private showAutoResolve: boolean = false;

  public render() {
    const { report, loading, success, error, onClick } = this.props;

    let buttonText = "Resolve";
    if (loading) {
      buttonText = "Loading...";
    } else if (success) {
      buttonText = "Resolved";
    } else if (error) {
      buttonText = "Error resolving";
    }

    const onClickNotPropogated = (e: React.MouseEvent) => {
      e.stopPropagation();
      onClick();
    };

    const button = (
      <ClickOutDetector onClickOut={this.closeAutoResolveCard}>
        <Layout position={Position.Relative}>
          <SplitButton
            data-track-click="report-resolve"
            dropdownButtonProps={{
              onClick: this.toggleAutoResolveCard,
              disabled: loading
            }}
            targetBlank
            disabled={loading}
            onClick={onClickNotPropogated}
            type={CoreButtonType.Primary}
          >
            <div className="report-button">
              <div className="button-text">{buttonText}</div>
            </div>
          </SplitButton>
          <AutoResolveCard report={report} show={this.showAutoResolve} onClose={this.closeAutoResolveCard} />
        </Layout>
      </ClickOutDetector>
    );

    if (success) {
      return <Tooltip label="Reassign to yourself to reopen">{button}</Tooltip>;
    }
    return button;
  }

  @action
  private closeAutoResolveCard = () => {
    this.showAutoResolve = false;
  };

  @action
  private toggleAutoResolveCard = () => {
    this.showAutoResolve = !this.showAutoResolve;
  };
}
