import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { Report } from "aegis/features/reports/models";
import * as Fragments from "aegis/fragments";
import { ResolveReportButtonComponent } from "./component";

export const RESOLVE_REPORT = gql`
  mutation resolveReport($input: ResolveReportInput!) {
    resolveReport(input: $input) {
      report {
        ...ReportFields
      }
    }
  }
  ${Fragments.ReportFields}
`;

type ResolveReportInput = {
  input: {
    id: string;
  };
};

type ResolveReportResponse = {
  resolveReport: {
    report: Report;
  };
};

export interface ResolveButtonProps {
  resolved?: boolean;
  report: Report;
  onResolve?: (report: Report) => void;
}

export class ResolveReportMutation extends Mutation<ResolveReportResponse, ResolveReportInput> {}

export const ResolveReportButton: React.SFC<ResolveButtonProps> = ({ resolved, report, onResolve }) => {
  return (
    <ResolveReportMutation
      mutation={RESOLVE_REPORT}
      onCompleted={data => {
        if (data && data.resolveReport && onResolve) {
          onResolve(data.resolveReport.report);
        }
      }}
      onError={error => console.error("Failed to resolve report: ", error.message)}
    >
      {(resolveReportQuery, { data, loading, error }) => {
        const onClick = () => {
          resolveReportQuery({ variables: { input: { id: report.id } } });
        };
        const button = (
          <ResolveReportButtonComponent
            report={report}
            loading={loading}
            error={error ? error.message : undefined}
            onClick={onClick}
            success={data != null || resolved}
          />
        );

        return button;
      }}
    </ResolveReportMutation>
  );
};
