import * as React from "react";

import { Report } from "aegis/features/reports";
import {
  Balloon,
  BalloonSize,
  Button,
  Color,
  CoreText,
  Interactable,
  Layout,
  Position,
  SplitButton,
  SVG,
  SVGAsset
} from "twitch-core-ui";

import { ClickOutDetector } from "aegis/functionality/components/click-out-detector";
import { UserRole } from "aegis/models";
import { User } from "aegis/models";
import { action, observable } from "mobx";
import { observer } from "mobx-react";

import "./styles.scss";

export interface Props {
  report: Report;
  onStrike?: (User?: User) => void;
}

export enum TestSelectors {
  PartnerStrikeExpandButton = "partner-strike-expand-button",
  PartnerStrikeButton = "partner-strike-button",
  PartnerConductFormButton = "partner-conduct-form-button",
  RegularStrikeButton = "regular-strike-button"
}

@observer
export class StrikeButton extends React.Component<Props> {
  @observable private showPartnerConductLink = false;

  public render() {
    const { report } = this.props;
    const resolved = report.status === "resolved";

    if (report.targetUser && report.targetUser.roles && report.targetUser.roles.includes(UserRole.Partner)) {
      // Partners need to be defaulted to partner conduct form
      return (
        <ClickOutDetector onClickOut={this.hidePartnerConductLink}>
          <Layout position={Position.Relative}>
            <SplitButton
              data-track-click="report-partner-conduct"
              dropdownButtonProps={{
                onClick: this.toggleShowPartnerConductLink,
                disabled: resolved || !this.props.report.targetUser
              }}
              targetBlank
              disabled={resolved}
              linkTo={`https://leviathan.internal.twitch.tv/reports/partnerconduct_report?reported_user=${
                report.targetUser.login
              }`}
              data-test-selector={TestSelectors.PartnerConductFormButton}
            >
              <CoreText color={resolved ? undefined : Color.Overlay}>Partner Conduct</CoreText>
            </SplitButton>
            <Balloon show={this.showPartnerConductLink} size={BalloonSize.ExtraSmall}>
              <Layout padding={{ x: 0.5, y: 0.5 }}>
                <Interactable
                  data-track-click="report-card-partner-strike"
                  onClick={this.onStrikeButtonClick}
                  blurAfterClick
                  data-test-selector={TestSelectors.PartnerStrikeButton}
                >
                  Strike
                </Interactable>
              </Layout>
            </Balloon>
          </Layout>
        </ClickOutDetector>
      );
    }

    // Return a simple button for non-partners
    return (
      <Button
        data-track-click="report-strike-submit"
        onClick={this.onStrikeButtonClick}
        disabled={resolved || !this.props.report.targetUser}
        data-test-selector={TestSelectors.RegularStrikeButton}
      >
        <div className="report-button strike-button">
          <div className="button-text">Strike</div>
          <div className="report-button-icon-overlay strike-icon-overlay">
            <SVG asset={SVGAsset.AngleRight} />
          </div>
        </div>
      </Button>
    );
  }

  private onStrikeButtonClick = () => {
    const { onStrike, report } = this.props;
    this.hidePartnerConductLink();
    if (onStrike) {
      onStrike(report.targetUser);
    }
  };

  @action
  private toggleShowPartnerConductLink = () => {
    this.showPartnerConductLink = !this.showPartnerConductLink;
  };

  @action
  private hidePartnerConductLink = () => {
    this.showPartnerConductLink = false;
  };
}
