import * as React from "react";

import { action, observable } from "mobx";
import { inject, observer } from "mobx-react";

import { AncestorName, AncestorNameContextProvider } from "aegis/context/ancestor";
import { Report } from "aegis/features/reports";
import { User } from "aegis/models";
import { AuthStoreProps } from "aegis/stores";
import { BorderRadius, Display, FlexDirection, JustifyContent, Layout, StyledLayout } from "twitch-core-ui";
import { ReportViewType } from "../../models";
import { ReportBanner } from "../report-banner";
import { ReportBody } from "../report-body";
import { ReportButtons } from "../report-buttons";
import { ReportHeader } from "../report-header";

import "./styles.scss";

export interface Props {
  report: Report;
  viewType: ReportViewType;
  disableExpand?: boolean;
  expanded?: boolean;
  collapseCard?: () => void;
  expandCard?: () => void;
  openStrikeForm: (User?: User) => void;
  toggleReassignForm?: () => void;
  onResolve?: () => void;
  hideReport?: () => void;
}

@inject("authStore")
@observer
export class ReportCard extends React.Component<Props & AuthStoreProps> {
  @observable hover: boolean = false;

  public render() {
    const { report, viewType, expanded, collapseCard, openStrikeForm, disableExpand } = this.props;
    const footer = !disableExpand && (
      <StyledLayout
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        padding={{
          top: expanded ? 0.5 : 0
        }}
        fullWidth
        borderLeft
        borderRight
        borderBottom
        borderRadius={{ bottomLeft: BorderRadius.Large, bottomRight: BorderRadius.Large }}
      >
        <div
          data-track-click="report-card-expand-toggle-footer"
          className={this.hover || expanded ? "report-view-expand-footer__visible" : "report-view-expand-footer"}
          onClick={expanded ? collapseCard : undefined}
        >
          {expanded ? "-" : "+"}
        </div>
      </StyledLayout>
    );

    if (expanded) {
      return (
        <StyledLayout
          className="report-view-card"
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          elevation={4}
        >
          <ReportBanner report={report} />
          <Layout>
            <ReportHeader report={report} expanded={expanded} />
            <ReportBody report={report} onStrikeFromUser={openStrikeForm} />
            <StyledLayout padding={{ x: 2 }} borderLeft borderRight>
              <ReportButtons
                report={report}
                viewType={viewType}
                onResolve={this.onResolve}
                onStrike={openStrikeForm}
                onReassign={this.onReassign}
                onInvestigate={this.onInvestigate}
                onCancelInvestigate={this.onCancelInvestigate}
                onHold={this.onHold}
                expanded
              />
            </StyledLayout>
          </Layout>
          {footer}
        </StyledLayout>
      );
    } else {
      return (
        <AncestorNameContextProvider ancestorName={AncestorName.CollapsedReportCard}>
          <div data-track-click="report-card-expand" onMouseEnter={this.onMouseEnter} onMouseLeave={this.onMouseLeave}>
            <StyledLayout
              className={"report-view-card"}
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              elevation={4}
            >
              <ReportBanner report={report} />
              <Layout className={"collapsed-report"}>
                <ReportHeader report={report} expanded={expanded}>
                  <div onClick={this.bubbleBurster}>
                    <ReportButtons
                      report={report}
                      viewType={viewType}
                      onReassign={this.onReassign}
                      onResolve={this.onResolve}
                      onInvestigate={this.onInvestigate}
                      onCancelInvestigate={this.onCancelInvestigate}
                    />
                  </div>
                </ReportHeader>
                {footer}
              </Layout>
            </StyledLayout>
          </div>
        </AncestorNameContextProvider>
      );
    }
  }

  private bubbleBurster = (e: React.MouseEvent<{}>) => {
    e.stopPropagation();
  };

  @action
  private onMouseEnter = () => {
    this.hover = true;
  };

  @action
  private onMouseLeave = () => {
    this.hover = false;
  };

  // HACK: force update report's corresponding values because apollo client updates the cache but not UI when
  // data is retrieved in a nested fashion (in our case paged response).
  // we can verify this is the case by clicking investigate/reassign button and see apollo cache value updates but UI stales
  //
  // https://github.com/apollographql/apollo-client/issues/3992#issuecomment-432330715
  // as of apollo-client 2.4.6 and apollo-in-memory-cache 1.3.10
  private onResolve = (updatedReport: Report) => {
    const { report, hideReport, collapseCard } = this.props;
    report.status = updatedReport.status;
    report.audits = updatedReport.audits;
    this.forceUpdate();

    if (hideReport) {
      hideReport();
    } else if (collapseCard) {
      collapseCard();
    }
  };

  private onReassign = (updatedReport: Report) => {
    const { report, toggleReassignForm, hideReport, collapseCard, authStore } = this.props;
    report.assignedto = updatedReport.assignedto;
    report.audits = updatedReport.audits;
    this.forceUpdate();

    if (
      !updatedReport.assignedto ||
      (authStore!.hasLoggedInUser && authStore!.loggedInUser!.ldap !== updatedReport.assignedto.ldap)
    ) {
      if (hideReport) {
        hideReport();
      } else if (collapseCard) {
        collapseCard();
      }
    }

    if (toggleReassignForm) {
      toggleReassignForm();
    }
  };

  private onInvestigate = (updatedReport: Report) => {
    const { report, expandCard } = this.props;
    report.assignedto = updatedReport.assignedto;
    report.audits = updatedReport.audits;
    this.forceUpdate();

    if (expandCard) {
      expandCard();
    }
  };

  private onCancelInvestigate = (updatedReport: Report) => {
    const { report, collapseCard, viewType, hideReport } = this.props;
    report.assignedto = updatedReport.assignedto;
    report.audits = updatedReport.audits;
    this.forceUpdate();

    if (hideReport && viewType === ReportViewType.Main) {
      hideReport();
    } else if (collapseCard) {
      collapseCard();
    }
  };

  private onHold = (updatedReport: Report) => {
    const { report, hideReport } = this.props;
    report.assignedto = updatedReport.assignedto;
    report.audits = updatedReport.audits;
    this.forceUpdate();
    if (hideReport) {
      hideReport();
    }
  };
}
