import * as React from "react";

import { Comment } from "aegis/features/reports";
import { TextWithLinks } from "aegis/functionality/components/text-with-links";
import { bullet } from "aegis/models";
import {
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Layout,
  StyledLayout,
  SVG,
  SVGAsset
} from "twitch-core-ui";

import "./styles.scss";

interface Props {
  comment: Comment;
  border?: boolean;
}

interface State {
  showFullText: boolean;
  needShowMoreButton: boolean;
}

export class CommentBox extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);

    this.state = {
      showFullText: false,
      needShowMoreButton: false
    };
  }

  public componentDidMount() {
    const commentDiv = document.getElementById(this.messageElementID());
    const fullTextDisplayed = !!commentDiv && commentDiv.offsetHeight < commentDiv.scrollHeight;
    this.setState({
      showFullText: !fullTextDisplayed,
      needShowMoreButton: fullTextDisplayed
    });
  }

  public render() {
    const { comment, border } = this.props;

    const ts = new Date(comment.createdAt);

    return (
      <Layout display={Display.Flex} padding={{ y: 0.5 }} fullWidth>
        <Layout padding={{ right: 1 }}>
          <StyledLayout color={Color.Alt2}>
            <SVG asset={SVGAsset.Conversations} />
          </StyledLayout>
        </Layout>
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <div id={this.messageElementID()} className="report-comment__message">
            <TextWithLinks
              className={this.state.showFullText ? "report-comment__message-full" : "report-comment__message-short"}
            >
              {comment.message}
            </TextWithLinks>
          </div>
          {this.state.needShowMoreButton && (
            <CoreLink
              data-track-click="report-comment-box-show-more"
              onClick={() =>
                this.setState({
                  showFullText: !this.state.showFullText
                })
              }
            >
              {this.state.showFullText ? "Show less" : "Show more"}
            </CoreLink>
          )}
          <StyledLayout borderBottom={border} padding={{ bottom: 0.5 }} display={Display.Flex} flexShrink={1}>
            <CoreText fontSize={FontSize.Size6} color={Color.Alt2}>
              {comment.createdBy.ldap} {bullet} {ts.toDisplayFormat()}
            </CoreText>
          </StyledLayout>
        </Layout>
      </Layout>
    );
  }

  private messageElementID() {
    return `report-comment__message-${this.props.comment.id}`;
  }
}
