import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { compose } from "react-apollo";

import { AncestorNameContext, AncestorNameContextMapper, withAncestorNameContext } from "aegis/context/ancestor";
import {
  ReportMutationContext,
  reportMutationContextMapper,
  withReportMutationContext
} from "aegis/features/report-view/components/mutations";

import { Button, ButtonSize, Layout, TextArea } from "twitch-core-ui";

export interface PublicProps {
  disabled?: boolean;
  onComplete: () => void;
}

@observer
export class ReportCommentInputComponent extends React.Component<
  PublicProps & ReportMutationContext & AncestorNameContext
> {
  @observable private comment: string = "";

  render() {
    const submitButton = (
      <Layout padding={{ top: 0.5, bottom: 1 }}>
        <Button data-track-click="report-comment-submit" onClick={this.onSubmit} size={ButtonSize.Small}>
          Comment
        </Button>
      </Layout>
    );
    const rows = this.comment === "" ? 1 : undefined;
    return (
      <Layout className="report-comment-input" flexGrow={1}>
        <TextArea
          placeholder="Add Comment"
          value={this.comment}
          onChange={this.updateComment}
          onKeyDown={this.onKeyDown}
          rows={rows}
          disabled={this.props.disabled}
        />
        {this.comment !== "" ? submitButton : null}
      </Layout>
    );
  }

  @action
  updateComment: React.FormEventHandler<HTMLTextAreaElement> = e => {
    this.comment = e.currentTarget.value;
  };

  @action
  onSubmit = () => {
    if (this.comment !== "") {
      this.props.createComment(this.comment, this.props.ancestorName, this.props.onComplete);
      this.comment = "";
    }
  };

  private onKeyDown = (e: React.KeyboardEvent<HTMLTextAreaElement>) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      this.onSubmit();
    }
  };
}

export const ReportCommentInput = compose(
  withAncestorNameContext<AncestorNameContext, PublicProps>(AncestorNameContextMapper),
  withReportMutationContext<ReportMutationContext, PublicProps>(reportMutationContextMapper)
)(ReportCommentInputComponent);
