import * as React from "react";

import { inject } from "mobx-react";

import { AncestorName } from "aegis/context/ancestor";
import { ImageDisplay, ImageStore, ImageUploader } from "aegis/features/images";
import { Comment, Report } from "aegis/features/reports";
import { OffsetPageInfo } from "aegis/models/offset_pageinfo";
import {
  AlignItems,
  Display,
  FlexDirection,
  FlexWrap,
  JustifyContent,
  Layout,
  Pagination,
  PaginationMouseEvent,
  StyledLayout
} from "twitch-core-ui";
import { ReportMutationContext, reportMutationContextMapper, withReportMutationContext } from "../mutations";
import { CommentBox } from "./comment-box";
import { ReportCommentInput } from "./comment-input";
import { COMMENTS_PER_PAGE } from "./container";

import "./styles.scss";

interface Props {
  screenshotStore?: ImageStore;
  report: Report;
  comments: Comment[];
  getCommentsForPage: (pageNumber: number) => void;
  pageInfo: OffsetPageInfo;
}

@inject("screenshotStore")
export class ReportCommentsViewComponent extends React.Component<Props & ReportMutationContext> {
  public render() {
    const { report, comments, getCommentsForPage, pageInfo } = this.props;
    const currentPage = pageInfo.offset / COMMENTS_PER_PAGE + 1;
    const totalPages = Math.ceil(pageInfo.total / COMMENTS_PER_PAGE);
    const onNewCommentCreated = () => getCommentsForPage(Math.ceil((pageInfo.total + 1) / COMMENTS_PER_PAGE));

    const commentView = comments.map((comment, index) => {
      return <CommentBox comment={comment} key={index} border={index === comments.length - 1 ? false : true} />;
    });
    return (
      <StyledLayout
        display={Display.Flex}
        alignItems={AlignItems.Center}
        flexWrap={FlexWrap.Wrap}
        justifyContent={JustifyContent.Center}
        padding={{ top: 1, x: 2 }}
        margin={{ top: 1 }}
      >
        <ImageDisplay store={this.props.screenshotStore!} />
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column} fullWidth>
          {commentView}
        </Layout>
        {totalPages > 1 && (
          <Layout fullWidth display={Display.Flex} justifyContent={JustifyContent.Start}>
            <Pagination
              currentPage={currentPage}
              totalPages={totalPages}
              onClickIndex={(e: PaginationMouseEvent<HTMLDivElement>): void => {
                getCommentsForPage(e.goToPage);
              }}
              onClickNext={() => getCommentsForPage(currentPage + 1)}
              onClickPrevious={() => getCommentsForPage(currentPage - 1)}
              nextPageButtonAriaLabel="Next Page"
              previousPageButtonAriaLabel="Previous Page"
            />
          </Layout>
        )}
        <Layout display={Display.Flex} fullWidth margin={{ top: 1 }}>
          <ReportCommentInput onComplete={onNewCommentCreated} />
          <Layout padding={{ right: 0.5 }} />
          <ImageUploader
            store={this.props.screenshotStore!}
            targetUserID={report.targetUser ? report.targetUser.id : undefined}
            imageUploaded={comment => {
              this.props.createComment(comment, AncestorName.ImageUploader, onNewCommentCreated);
            }}
          />
        </Layout>
      </StyledLayout>
    );
  }
}

export const ReportCommentsView = withReportMutationContext<ReportMutationContext, Props>(reportMutationContextMapper)(
  ReportCommentsViewComponent
);
