import * as React from "react";

import gql from "graphql-tag";

import { CommentConnection, Report } from "aegis/features/reports";
import { CommentFields } from "aegis/fragments";
import { ReportCommentsView } from "./component";

import { Query } from "aegis/functionality/apollo-wrapper";
import { Placeholder } from "twitch-core-ui";

export const COMMENTS_PER_PAGE = 5;
export const FETCH_REPORT_COMMENTS = gql`
  query reportComments($id: ID!, $offset: Int) {
    report(id: $id) {
      id
      comments(limit: ${COMMENTS_PER_PAGE}, offset: $offset) {
        comments {
          ...CommentFields
        }
        pageInfo {
          offset
          total
        }
      }
    }
  }

  ${CommentFields}
`;

type CommentsQueryInput = {
  id: string;
  offset: number;
};

type CommentsResponse = {
  report?: {
    comments?: CommentConnection;
  };
};

export type Props = {
  report: Report;
};

export class ReportWithCommentsQuery extends Query<CommentsResponse, CommentsQueryInput> {}

export const ReportComments: React.SFC<Props> = ({ report }) => {
  return (
    <ReportWithCommentsQuery
      displayName="ReportComments"
      query={FETCH_REPORT_COMMENTS}
      variables={{ id: report.id, offset: 0 }}
    >
      {({ loading, error, data, refetch }) => {
        if (loading) {
          return (
            <>
              <Placeholder width={200} lineCount={3} />
              <Placeholder width={200} lineCount={3} />
              <Placeholder width={200} lineCount={3} />
            </>
          );
        }

        if (error || !data || !data.report || !data.report.comments) {
          console.error("Failed to fetch comments", error);
          return (
            <div>
              Error: Failed to load comments. If this error persists through a refresh, please reach out to Safety
              Oncall.
            </div>
          );
        }

        const connection = data.report.comments;
        const getCommentsForPage = (pageNumber: number) => {
          refetch({ id: report.id, offset: COMMENTS_PER_PAGE * (pageNumber - 1) });
        };

        return (
          <ReportCommentsView
            report={report}
            getCommentsForPage={getCommentsForPage}
            comments={connection.comments}
            pageInfo={connection.pageInfo}
          />
        );
      }}
    </ReportWithCommentsQuery>
  );
};
