import * as React from "react";

import {
  ChatColumnNameOverrides,
  CollapsibleMessageLog,
  HighlightedRows,
  Message,
  MessageColumn
} from "aegis/features/messages";
import { HighlightType } from "aegis/features/messages/components/message-log-row";
import { ChatReport } from "aegis/features/reports";
import { VodPopoutButton } from "aegis/features/vod-popout-button";
import { Button, ButtonType, Color, CoreText, Display, Layout, SVG, SVGAsset } from "twitch-core-ui";

interface Props {
  report: ChatReport;
}

export enum TestSelectors {
  MessageEntries = "message-entries",
  ErrorMessage = "error-message"
}

export class ChatEmbed extends React.Component<Props> {
  public render() {
    const { report } = this.props;

    const entries = report.entries && report.entries.edges.map(m => m.node);

    if (!entries || entries.length === 0) {
      if (report.channelID) {
        const createdAt = new Date(report.createdAt);
        const chatLogsLink = `/chatlogs/?channelids=${report.channelID}&before=${createdAt.getTime()}`;
        return (
          <Button
            data-track-click="report-chat-embed-see-chatlogs"
            type={ButtonType.Hollow}
            linkTo={chatLogsLink}
            targetBlank
            tabIndex={-1}
            fullWidth={false}
          >
            <Layout display={Display.Flex}>
              See chatlogs for channel at report creation time
              <SVG asset={SVGAsset.Popout} />
            </Layout>
          </Button>
        );
      }

      return (
        <CoreText color={Color.Error} data-test-selector={TestSelectors.ErrorMessage}>
          Couldn't retrieve chat logs related to this report
        </CoreText>
      );
    }

    const messageLogSummary: Message[] = [];
    const highlights: HighlightedRows = {};
    entries.forEach(entry => {
      if (entry.id === report.messageID) {
        messageLogSummary.push(entry);
        highlights[entry.id] = HighlightType.Main;
      } else if (entry.from && report.targetUser && entry.from.id === report.targetUser.id) {
        highlights[entry.id] = HighlightType.Alt;
      }
    });

    return (
      <CollapsibleMessageLog
        entries={entries}
        columnNameOverride={ChatColumnNameOverrides}
        highlights={highlights}
        messageLogSummary={messageLogSummary}
        chatColumns={[MessageColumn.Time, MessageColumn.From, MessageColumn.Message, MessageColumn.Actions]}
        additionalActionFactory={this.additionalActionsFactory}
        data-test-selector={TestSelectors.MessageEntries}
      />
    );
  }

  private additionalActionsFactory = (message: Message) => {
    if (!message.target) {
      return [];
    }

    const timestamp = parseInt(message.sentTimestamp, 10) / 1000; // ms to seconds

    return [<VodPopoutButton key={message.id} channelID={message.target.id} timestamp={timestamp} />];
  };
}
