import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { Router } from "react-router-dom";

import { DELETE_CLIPS, DeleteClipsButton } from "aegis/features/delete-clips-button";
import * as ReportModels from "aegis/features/reports/models";
import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { createHistory } from "aegis/functionality/utils/contrib/twilight/history";
import { mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button } from "twitch-core-ui";
import { ClipEmbed, TestSelectors } from "./clip-embed";
import { NewClipReport } from "./testutil";

Enzyme.configure({ adapter: new Adapter() });

const mockDeleteClipSuccess = mockSuccessResponse(
  DELETE_CLIPS,
  { slugs: { slugs: ["slug"] } },
  {
    data: {
      deleteClips: {
        success: [true],
        __typename: "ClipReport"
      }
    }
  }
);

function setupMount(overrides?: Partial<ReportModels.ClipReport>) {
  const report = NewClipReport(overrides ? overrides : {});
  return Enzyme.mount(
    <Router history={createHistory()}>
      {wrapWithGQLProvider(<ClipEmbed report={report} />, [mockDeleteClipSuccess]).wrappedComponent}
    </Router>
  );
}

describe("ClipEmbed", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders ui parts", () => {
    const wrapper = setupMount();
    expect(wrapper.find({ "data-test-selector": TestSelectors.Iframe })).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.VodURLButton })).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.ClipOverviewButton })).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.DeleteClipButton })).toBeTruthy();
  });

  it("does not show clip embed if clip slug is not found", () => {
    const wrapper = setupMount({ slug: undefined });
    expect(wrapper.find({ "data-test-selector": TestSelectors.ClipNotFoundText }).length).toBeGreaterThan(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.Iframe })).toHaveLength(0);
    expect(wrapper.text()).toContain("Clip Not Found / Deleted");
  });

  it("disables Vod URL button if vodURL is undefined", () => {
    const wrapper = setupMount({ vodURL: undefined });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.VodURLButton })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables Delete Clip Button on successful deletion", async () => {
    const wrapper = setupMount();

    expect(wrapper.find(DeleteClipsButton).prop("disabled")).toBe(false);
    wrapper
      .find(DeleteClipsButton)
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");

    await waitSomeMs();
    wrapper.update();

    expect(wrapper.find(DeleteClipsButton).prop("disabled")).toBe(true);
    expect(wrapper.find(DeleteClipsButton).text()).toContain("Clip Deleted");
    expect(wrapper.find({ "data-test-selector": TestSelectors.ClipNotFoundText })).toHaveLength(0);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorText })).toHaveLength(0);
  });
});
