import * as React from "react";

import { observable } from "mobx";
import { observer } from "mobx-react";

import { DeleteClipsButton, DeleteClipsResponse } from "aegis/features/delete-clips-button";
import { ClipReport } from "aegis/features/reports";
import { Button, ButtonType, Color, CoreText, Display, FormGroup, Layout } from "twitch-core-ui";
import { StrikeableClipCurator } from "./components/strikeable-clip-curator";

export enum TestSelectors {
  Iframe = "iframe",
  ClipOverviewButton = "clip-overview-button",
  VodURLButton = "vod-url-button",
  DeleteClipButton = "delete-clip-button",
  ClipNotFoundText = "clip-not-found",
  ErrorText = "clip-error-text"
}
export interface PublicProps {
  report: ClipReport;
}

type Props = PublicProps;

@observer
export class ClipEmbed extends React.Component<Props> {
  @observable private deleted: boolean = false;
  @observable private error: Error | null = null;

  public render() {
    const { id, clipURL, vodURL, clipOverviewURL, curator, slug } = this.props.report;

    if (!slug) {
      return (
        <CoreText color={Color.Error} data-test-selector={TestSelectors.ClipNotFoundText}>
          Clip Not Found / Deleted
        </CoreText>
      );
    }

    return (
      <Layout display={Display.Flex}>
        <div>
          <FormGroup id={`clip-embed-${id}`} label="Clip">
            <iframe
              width="480"
              height="270"
              allowFullScreen
              src={clipURL + "&autoplay=false"}
              data-test-selector={TestSelectors.Iframe}
            />
          </FormGroup>
          <Button
            data-track-click="report-clip-embed-clips-overview"
            type={ButtonType.Hollow}
            linkTo={clipOverviewURL}
            targetBlank={true}
            disabled={!clipOverviewURL}
            data-test-selector={TestSelectors.ClipOverviewButton}
          >
            <div className="report-button">
              <div className="button-text">Clip Overview</div>
            </div>
          </Button>
          <Button
            data-track-click="report-clip-embed-vod-url"
            type={ButtonType.Hollow}
            linkTo={vodURL}
            targetBlank={true}
            disabled={!vodURL}
            data-test-selector={TestSelectors.VodURLButton}
          >
            <div className="report-button">
              <div className="button-text">VOD</div>
            </div>
          </Button>
          <DeleteClipsButton
            slugs={[slug]}
            type={ButtonType.Alert}
            onCompleted={this.onCompleted}
            onError={this.onError}
            disabled={!slug || this.deleted}
            data-test-selector={TestSelectors.DeleteClipButton}
          >
            <div className="report-button">
              <div className="button-text">{this.deleted ? "Clip Deleted" : "Delete Clip"}</div>
            </div>
          </DeleteClipsButton>
          {this.error && (
            <CoreText color={Color.Error} data-test-selector={TestSelectors.ErrorText}>
              {this.error.toString()}
            </CoreText>
          )}
        </div>
        <Layout padding={{ x: 2 }}>
          <CoreText bold>Clipper</CoreText>
          {curator && <StrikeableClipCurator user={curator} />}
        </Layout>
      </Layout>
    );
  }

  private onError = (error: Error) => {
    this.error = error;
  };

  private onCompleted = (response: DeleteClipsResponse) => {
    this.deleted = response.deleteClips.success[0];
  };
}
